/*
  These routines handle the reading & writing of OS9 file structures

  Some of these are a lot like normal C file I/O routines.
*/


int OS9FSeek(OS9FILE *fd, unsigned long pos)
/*
  This routine positions the file pointer at the desired location.
  It does handle segments.
*/
{unsigned long cursize,offset;
if (options.debug) printf("File seek to FPos %lu\n",pos);

if (pos > fd->fd_fsize) return FALSE;

fd->activeseg=0;
fd->curpos   =fd->fd_seg[0].baddr;
fd->segbytes =fd->fd_seg[0].bsize;
fd->bytesleft=fd->fd_fsize;


cursize=0L;
while (pos > (cursize + fd->segbytes))
  {cursize += fd->segbytes;
   fd->bytesleft -= fd->segbytes;
   if (!GetNextSeg(fd)) return FALSE;
  }

offset=pos - cursize;
fd->segbytes  -= offset;
fd->curpos    += offset;
fd->bytesleft -= offset;
return TRUE;
}

/*
===================================================================
This is the routine that you actually call to read the next 'x'
bytes from a file descriptor.  It handles all the details.
It's a lot like normal fread()
===================================================================
*/

unsigned int OS9FRead(OS9FILE *fd, void *buffer, unsigned int lenwanted)
/*
  Try to read 'lenwanted' bytes from the file 'fd'.  If no more
  bytes to read, 0 (FALSE) is returned, otherwise it returns the
  number of bytes read.

  This is level 3 of the sector input routines.  This is the one
  that knows about segments and so on.  If you need to read from a
  file (or anything like that, such as a directory), this is the one
  you should call.  You may position the file using OS9FSeek() first
  if needed.
*/
{unsigned int BytesAvail;
 char *bufptr= (char *) buffer;
 unsigned int len=lenwanted;

if (options.debug) printf("OS9FRead %u bytes\n",lenwanted);

while (len != 0)
  {
   if (fd->bytesleft == 0L) break;  /* any more bytes left in file? */
   else if (fd->segbytes==0L)  /* any more bytes left in the segment? */
          {if (!GetNextSeg(fd)) return FALSE;}  /* no more segments either */
   else {
         BytesAvail=len;
         if (BytesAvail > fd->segbytes)
            BytesAvail=(unsigned int) fd->segbytes;
         if (BytesAvail > fd->bytesleft)
            BytesAvail=(unsigned int) fd->bytesleft;

         ReadBytes(fd->curpos,bufptr,BytesAvail);
         fd->curpos    += BytesAvail;
         fd->segbytes  -= BytesAvail;
         fd->bytesleft -= BytesAvail;
         bufptr += BytesAvail;
         len -= BytesAvail;
        }
  }
return (lenwanted - len);
}

/*
===================================================================
This is the routine that you actually call to write the next 'x'
bytes to a file descriptor.  It handles all the details.
It's a lot like normal fwrite()
===================================================================
*/

unsigned int OS9FWrite(OS9FILE *fd, void *buffer, unsigned int lenout)
/*
  Try to write 'lenwanted' bytes to the file 'fd'.
  
  Returns the number of bytes written unless an error occurs, in which
  case it returns 0 (FALSE)

  This is level 3 of the sector output routines.  This is the one
  that knows about segments and so on.  If you need to write to a
  file (or anything like that, such as a directory), this is the one
  you should call.  You may position the file using OS9FSeek() first
  if needed.
*/
{unsigned int BytesAvail;
 char *bufptr= (char *) buffer;
 unsigned int len=lenout;

if (options.debug) printf("OS9FWrite %u bytes\n",lenout);

while (len != 0)
  {
   if (fd->segbytes==0L)       /* any more bytes left in the segment? */
       {if (!GetNextSeg(fd))   /* no more segments either */
           if (!ExtendFile(fd,len)) /* can't extend it either! */
              return FALSE;
        continue;
       }
   else {
         BytesAvail=len;
         if (BytesAvail > fd->segbytes)
            BytesAvail=(unsigned int) fd->segbytes;
         if (BytesAvail > fd->bytesleft)
            BytesAvail=(unsigned int) fd->bytesleft;

         WriteBytes(fd->curpos,bufptr,BytesAvail);
         fd->curpos    += BytesAvail;
         fd->segbytes  -= BytesAvail;
         fd->bytesleft -= BytesAvail;
         bufptr += BytesAvail;
         len -= BytesAvail;
        }
  }
return (lenout - len);
}



/*
=================================================================
These are the routines that are used to read the OS9 structures
They perform the structure conversion for you.
=================================================================
*/

int ReadDirEntry(OS9FILE *fd,struct DirEntry *direntry)
/*
  Read OS9 s_dirent in and convert it to PC version
  Since OS9FRead can read less than the number we want, we have
  to check it.  Anything less is considered a failure.
*/
{struct s_dirent OS9_dir;int ok,bytesread;
if (options.debug) printf("ReadDirEntry\n");
bytesread=OS9FRead(fd,&OS9_dir,sizeof(OS9_dir));
ok=(bytesread == sizeof(OS9_dir));
if (ok) *direntry=CvtDirEntry2PC(&OS9_dir);
return ok;
}

OS9FILE OS9FOpen(long pos)
/*
  Read in OS9 file descriptor and convert it to PC version
*/
{struct s_filedesc OS9_fdesc;OS9FILE PC;
if (options.debug) printf("OS9FOpen\n");
ReadBytes(pos,&OS9_fdesc,sizeof(OS9_fdesc));
PC=CvtFDesc2PC(&OS9_fdesc);
OS9FSeek(&PC,0L);
PC.homeptr=pos;
return PC;
}


unsigned long SearchDir(OS9FILE dirdesc,char *FName,struct DirEntry *dirent)
/*
  Search the current directory for FName.  If you'll notice, the
  file descriptor for the directory is passed by value.  This way,
  the calling one won't be modified by the ReadDirEntry().
*/
{unsigned long curpos;
if (options.debug) printf("SearchDir\n");
curpos=dirdesc.curpos;
while (ReadDirEntry(&dirdesc,dirent))
  {if (CmpStr(FName,dirent->dir_name)) return curpos;
   curpos=dirdesc.curpos;
  }
return 0L;
}


struct DirEntry WalkDirPath(void)
/*
  Walk the OS9 directory structure to the target sub-dir
*/
{OS9FILE fdesc,dirdesc;
 struct DirEntry direntry;
 char fname[64];
 unsigned long found;

direntry=rootdirentry;
dirdesc=OS9FOpen(diskinfo.dd_dir);

if (options.debug) printf("Starting to walk directory\n");

while (TRUE)
  {
   if (!PeelPathName(fname)) break; /* end of path */

if (options.debug) printf("Walking, looking for %s\n",fname);
   found=SearchDir(dirdesc,fname,&direntry);

   if (found)
     {
if (options.debug) printf("%s was found\n",fname);
      fdesc=OS9FOpen(direntry.dir_addr);
      if ((fdesc.fd_att & 0x80) == 0x80) dirdesc=fdesc;
      else {  /* filename found, but isn't a directory! */
            fprintf(stderr,"%s in path isn't a directory\n",fname);
            QuitProgram(ENOPATH);
           }
     }
   else {fprintf(stderr,"%s is not found in path search\n",fname);
         QuitProgram(ENOPATH);
        }
  }

return direntry;
}

int ExtendFile(OS9FILE *fd,unsigned long amount)
/*
  This will extend a file's size by 'amount' bytes.  It does write the
  DAM and descriptor back to the disk.

  Also, it positions the file pointer to the end of the original file
  size.  This lets you 'append' to the file (and just 'continue' writing
  without actually having to know the sector LSN.
*/
{unsigned long wantsize,wantsectors,needsectors,cursectors,allocsize;
 struct s_filedesc fdOS9;
 struct SegList lastseg;

if (options.debug) printf("Extending file by %u bytes\n",amount);

lastseg=fd->fd_seg[fd->activeseg];
allocsize=GetFileAllocSize(fd);
wantsize=fd->fd_fsize + amount;
OS9FSeek(fd,fd->fd_fsize);  /* move pointer to end of file */

if (wantsize <= allocsize)
   {if (options.debug)
      printf("Request for %u more bytes is satisfied within current allocation\n");
   }
else {  /* We'll have to add sectors */
      cursectors = (allocsize + SectorSize - 1) / SectorSize;
      wantsectors = (wantsize + SectorSize - 1) / SectorSize;
      needsectors = wantsectors - cursectors;

      if (options.debug)
        {printf("Current allocated size %lu bytes. Want %lu more bytes.\n",
                fd->fd_fsize,amount);
         printf("Currnt Sect %lu. Need %lu more\n",cursectors,needsectors);
        }

      if ((needsectors+diskinfo.sectorsused) > diskinfo.dd_tot)
         {if (options.debug) printf("Not enough free space on disk\n");
          return FALSE;
         }


      while (needsectors != 0L)
        {
         if (options.debug > 2) {printf("Before\n");DumpSeg(fd);}
         if (!AddSector(fd))
          {fprintf(stderr,"Unable to allocate more disk sectors\n");
           QuitProgram(0);
          }
         if (lastseg.bsize != fd->fd_seg[fd->activeseg].bsize) 
            {fd->segbytes+=SectorSize;lastseg=fd->fd_seg[fd->activeseg];}
         needsectors--;
         if (options.debug > 2) {printf("After\n");DumpSeg(fd);}
        }
     }


fd->fd_fsize=wantsize;
fd->bytesleft += amount;

/* write the updated DAM and file descriptor back to the disk. */
fdOS9=CvtFDesc2OS9(fd);
WriteBytes(fd->homeptr,&fdOS9,sizeof(fdOS9));
WriteDAM();

return TRUE;
}


