/*
   These routines handle lower level disk accesses.

   Currently, this doesn't allow the disk to have different number of
   sectors per track.  Traditional/classic OS9 let's track 0 be in single
   density and have fewer tracks than the rest of the disk.

   The Turbo C docs say that BIOSDISK() returns ah in the upper byte
   and al in the lower byte.  It doesn't.  Instead, the upper byte is
   always 0 and the lower byte contains ah (the error code).  I checked
   the disassembly of the biosdisk module.  TC's docs are wrong.

*/

/* The sector cache */
long cachedlsn=-1L;
unsigned char _CaChEdSeCt[MaxSectSize];

int ReadTheDisk(long lsn,int nsects, unsigned char *buffer)
/* 
   Read sector LSN
   This will be changed to fit MS-DOS's absolute sector I/O rather than
   OS9's byte offset access to disk
   This is the only function that actually physically reads the disk.

   This routine is capable of reading multiple sectors at a time.
   HOWEVER, it's up to you to actually determine that there are that
   many sectors left on the track!  See, it uses BIOSDISK() and that
   can only read multiple sectors when there are that many left on
   the track and on that head.  It doesn't wrap.  So, be carefull
   and use at your own risk.

   You really shouldn't call this directly since little error checking
   is done and no retries are performed.

   This is level 0 of the disk input routine.  All it knows how to do
   is low level sectior input.

*/
{int sect,track,head;
int lenread;
int failed;

/* Invalidate the cache if it's more than 1 sector */
if (nsects != 1) cachedlsn=-1L;

if (lsn==cachedlsn)
  {memcpy(buffer,_CaChEdSeCt,SectorSize);
   if (options.debug >1 ) printf("Cache hit at lsn: %lu\n",lsn);
   return TRUE;
  }

failed=FALSE;

if (options.virtdisk)
  {failed=fseek(options.pcvirtfile,lsn*SectorSize,SEEK_SET);
   if (failed)
     {if (options.debug)
        fprintf(stderr,"Unable to seek to %lu\n",lsn*SectorSize);
     }
   else
     {if (options.debug > 2) printf("ReadTheDisk LSN=%lu\n",lsn);
      lenread=fread(buffer,SectorSize,nsects,options.pcvirtfile);
      if (lenread != nsects) failed=TRUE;
     }
  }
else {
      LSN2Floppy(lsn,&track,&sect,&head);

      if (options.debug > 2)
        {printf("ReadTheDisk LSN=%lu\n",lsn);
         printf("Track=%d Sect=%d Head=%d\n",track,sect,head);
        }

      failed=biosdisk(_DISK_READ,options.drivenum,head,track,sect,nsects,buffer);

      if (failed)
        {if (options.debug > 1)
            {fprintf(stderr,"Unable to read sector.  Error: %d\n",failed);
             fprintf(stderr,"drive: %d\n",options.drivenum);
            }
        }
     }

if (options.debug > 1)
  {printf("failed: %d, nsects: %d, cachedlsn: %lu\n",failed,nsects,cachedlsn);}

if ((failed) || (nsects !=1)) /* if failed read or multi-sector read, */
   cachedlsn=-1L;             /* then clear the cache */
else          /* else it was a successful single sector read, */
 {memcpy(_CaChEdSeCt,buffer,SectorSize);  /* so cache it. */
  cachedlsn=lsn;
 }

return (failed==0);
}

int WriteTheDisk(long lsn,int nsects,unsigned char *buffer)
/*
   Write sector LSN
   This will be changed to fit MS-DOS's absolute sector I/O rather than
   OS9's byte offset access to disk
   This is the only function that actually physically writes the disk.

   This routine is capable of writing multiple sectors at a time.
   HOWEVER, it's up to you to actually determine that there are that
   many sectors left on the track!  See, it uses BIOSDISK() and that
   can only read multiple sectors when there are that many left on
   the track and on that head.  It doesn't wrap.  So, be carefull
   and use at your own risk.

   It does not cache write sectors.

   You really shouldn't call this directly since little error checking
   is done and no retries are performed.

   This is level 0 of the disk output routine.  All it knows how to do
   is low level sectior output.

*/
{int sect,track,head;
int lenwrote;
int failed;

/*
  Invalidate the cache.

  We could play tricks, like caching writes, or write through caches,
  but it works just as well to simply totally invalidate the cache.
  And there is less chance for error.
*/
cachedlsn=-1L;

if (options.virtdisk)
  {failed=fseek(options.pcvirtfile,lsn*SectorSize,SEEK_SET);
   if (failed)
     {if (options.debug)
         fprintf(stderr,"Unable to seek to %lu\n",lsn*SectorSize);
      return FALSE;
     }
   if (options.debug > 2) printf("WriteTheDisk LSN=%lu\n",lsn);
   lenwrote=fwrite(buffer,SectorSize,nsects,options.pcvirtfile);
   fflush(options.pcvirtfile);
   if (lenwrote != nsects) return FALSE;
  }
else {
      LSN2Floppy(lsn,&track,&sect,&head);

      if (options.debug > 2)
        {printf("WriteTheDisk LSN=%lu\n",lsn);
         printf("Track=%d Sect=%d Head=%d\n",track,sect,head);
        }

      failed=biosdisk(_DISK_WRITE,options.drivenum,head,track,sect,nsects,buffer);

      if (failed)
        {if (options.debug > 1)
            {fprintf(stderr,"Unable to write sector.  Error: %d\n",failed);
             fprintf(stderr,"drive: %d\n",options.drivenum);
            }
         return(FALSE);
        }
     }
return(TRUE);
}

void LSN2Floppy(long lsn,int *track, int *sect, int *head)
/* Convert LSN into floppy positioning */
/* If I need to deal with tracks having different number of sectors,
   here is where I'd do it.
*/
{
*track= (int) (lsn / diskinfo.cylsectors);
*sect=((int) (lsn % diskinfo.cylsectors));  /* 0 based */
*head=*sect / diskinfo.sectors;
*sect=(*sect % diskinfo.sectors) + 1;  /* cvt to 1 based */
}


void ResetDisk(void)
/*
  Tell the disk circuitry to reset itself
*/
{unsigned int result;
if (options.virtdisk)
   {
    if (options.pcvirtfile != NULL) rewind(options.pcvirtfile);
   }

if (options.drivenum != -1)
   {
    result=biosdisk(_DISK_RESET,options.drivenum,0,0,0,0,NULL);
    if ((result & 0xff00) !=0)
      {fprintf(stderr,"Error reseting disk %d\n",result);
       QuitProgram(EFAULT);
      }
   }
}

void FinishedDisk( void )
/*
  This has to be done or you won't be able to access the
  drive from MS-DOS
*/
{char far *(far *vect78h);char far * bpsvec;

if (options.virtdisk)
  {
   if (options.pcvirtfile != NULL) fclose(options.pcvirtfile);
  }

if (options.drivenum != -1)
     {
      vect78h=(char far *(far *))(0x78);
      bpsvec=3 + *vect78h;
      *bpsvec=(char) 2;  /* Switch back to regular DOS 512 bytes/sect */
     }
}

void PrepOS9Disk(void)
/*
  do what you need to be able to read the disk by sectors
  Tell MS-DOS that we are using 256 byte sectors instead of 512
  0=128, 1=256, 2=512, 3=1024
*/
{char far *(far *vect78h);char far * bpsvec;char *op;

if (options.virtdisk) /* r+b varies from compiler to compiler */
   {if (options.put) op="r+b"; else op="rb"; /* for protection */
    options.pcvirtfile=fopen(options.vdiskname,op);
    if (options.pcvirtfile == NULL)
      {fprintf(stderr,"Unable to open virtual disk: %s\n",options.vdiskname);
       QuitProgram(ENOFILE);
      }
   }

if (options.drivenum != -1)
     {
      vect78h=(char far *(far *))(0x78);
      bpsvec=3 + *vect78h;
      *bpsvec=(char) DosBPS;
     }
}

