/*****************************************************************************
* CINESIM.C																						  *
*																									  *
* Cinematronics Simulator.																	  *
*																									  *
* This program is used to simulate many of the signals that appear in the	  *
* Cinematronics CPU schematic.  In order to make use of this program			  *
* a schematic is mandatory.																  *
*																									  *
* Author:  Zonn Moore																		  *
* Version: 1.0																					  *
* Date:    01/31/97																			  *
*																									  *
* Released to public domain by the author, distribute freely.					  *
*****************************************************************************/
#include	<stdio.h>
#include	<stdlib.h>
#include	<conio.h>
#include	<ctype.h>
#include	<io.h>
#include	<fcntl.h>
#include	<sys\types.h>
#include	<sys\stat.h>

#define	JREG		16					// address of any jump

// Define some common data types.

typedef unsigned char	uchar;
typedef unsigned int		uint;

uchar	Even[16], Odd[16];			// simulate up to 32 instructions

// Prom storage

uchar	C14[32], D14[32], E14[32], F14[256], E8[32], J14[32];

/*****************************************************************************
* Read in PROMs from image files.														  *
*****************************************************************************/
void readProms( void)
{
	int	inFile, err;

	// Who knows what we might need later, lets just read in all the PROMs
	// now.

	inFile = _open( "prom.c14", O_BINARY | O_RDONLY);
	err = _read( inFile, C14, sizeof( C14));

	if (err == -1)
	{	perror( "prom.c14");
		exit( 1);
	}
	_close( inFile);

	inFile = _open( "prom.d14", O_BINARY | O_RDONLY);
	err = _read( inFile, D14, sizeof( D14));

	if (err == -1)
	{	perror( "prom.d14");
		exit( 1);
	}
	_close( inFile);

	inFile = _open( "prom.e14", O_BINARY | O_RDONLY);
	err = _read( inFile, E14, sizeof( E14));

	if (err == -1)
	{	perror( "prom.e14");
		exit( 1);
	}
	_close( inFile);

	inFile = _open( "prom.f14", O_BINARY | O_RDONLY);
	err = _read( inFile, F14, sizeof( F14));

	if (err == -1)
	{	perror( "prom.f14");
		exit( 1);
	}
	_close( inFile);

	inFile = _open( "prom.e8", O_BINARY | O_RDONLY);
	err = _read( inFile, E8, sizeof( E8));

	if (err == -1)
	{	perror( "prom.e8");
		exit( 1);
	}
	_close( inFile);

	inFile = _open( "prom.j14", O_BINARY | O_RDONLY);
	err = _read( inFile, J14, sizeof( J14));

	if (err == -1)
	{	perror( "prom.j14");
		exit( 1);
	}
	_close( inFile);
}

// Return the value of bit number 'bitv' in mask 'x'

#define	bit( x, bitv)	((x & (1 << bitv)) ? 1 : 0)

// Enums used to associate output lines of PROMs, Latches and Flip/Flops
// with bit values.

enum { O1, O2, O3, O4, O5, O6, O7, O8};
enum { QA, QB, QC, QD};
enum { Q5, _Q6, Q3, _Q2};	// 'Q' and associate pin number

// Every latch, signal, and flip/flop we want to study is given a name.

uint	Ar, Jr;				// Address (PC) and 'J registers
uchar	Ir, Dr, U9T9;		// Instruction latch and Data latch, Temp hold latches

/*
Define:
   E10      = Opcode cycle, 4bit counter.
   A8       = A/B register latch.
   B8       = Read / Write memory latch.
   I14      = Wait state generator, 4bit latch.
   J14_adr  = Temp reg used to hold 'J14's address.
   Prom_adr = Temp reg used to hold the address of the microcode PROMs.
*/

uchar	E10, A8, B8, I14, J14_adr, Prom_adr;

/*
When a name is given to a signal in the schematic, that name will be
signal name, using the following rules:

   All characters not allowed in a label will be converted to a '_'
   Ex: S0* becomes S0_

   All negative logic values are proceeded with a '_'.

If a signal is needed for program flow, or to be printed during
simulation, and no name is given the signal, the follow rule applies:

   Take the name of the IC that generates the signal, followed by
   an underscore, followed by the pin number of the signal.
   Ex: Output pin 6 of IC C12 would be named 'C12_6'.
*/

uchar	Adv, _Acc_, _Lir_, _Ldr_, Ld_ps, Ld_im, _Vector, _NVector, _Par2;
uchar	_ColdStart, S_write_ws, Write_ext, S1, S0_, X_arith, Sr24, Y;
uchar	_Wsa_, Read, Ds3, Page, _Frame, _Jump, Look_up, Multiply;
uchar	_Indr_adr, Sq0, Sq1, _0, _1, Acc_pm, L_ff, _L_ff, Lir, _Lir, Ldr;
uchar	_Ldr, A0, _Branch, _Lpa, Ipc, A0_, _Delay, _Latch, _Ready, Ready;
uchar	J14_o1, J14_o2, J14_o5, H14_1, A10_8, D12_6, B10_11, F12_6, Prog_page;
uchar	I4_ready, Loadtimer, Shifted, D10_10, A14_3, B12_8, Acc;
uchar	S0, A14_6, Par1, Ds0, Ds1, Ds1_, Ds2, Indr, Secondary, _Secondary;
uchar	R_w, _Proceed,	K4_10, Xorp, Xors, C12_12,	C12_6, C12_8, _Reset;
uchar	_Jmpsel;

// A list of names and pointers to the signals to be displayed.
//
// Descriptions and signals can be added or removed from this array
// at will.

struct
{	char	*name;			// name of signal
	uchar	*value;			// pointer to signals value
} Signals[] =
{
	" LD PG PS", 		&Ld_ps,
	" LD PG IMID",		&Ld_im,
	"~VECTOR",			&_Vector,
	"~NRM VECTOR",		&_NVector,
	"~PAR2",				&_Par2,
	"~COLD START",		&_ColdStart,
	" S-WRITE WS",		&S_write_ws,
	" WRITE_EXT",		&Write_ext,
	" S1",				&S1,
	" S0*",				&S0_,
	" X_ARITH",			&X_arith,
	" SR24",				&Sr24,
	" Y",					&Y,
	" DS0",				&Ds0,
	" DS1*",				&Ds1_,
	"~WSA*",				&_Wsa_,
	"~READ",				&Read,
	" DS3",				&Ds3,
	" PAGE",				&Page,
	"~FRAME",			&_Frame,
	"~JUMP",				&_Jump,
	" LOOK_UP",			&Look_up,
	" MULTIPLY",		&Multiply,
	"~INDR ADR",		&_Indr_adr,

	" ADV",				&Adv,
	"~ACC*", 			&_Acc_,
	"~LIR*", 			&_Lir_,
	"~LDR*",				&_Ldr_,

	" SQ0",				&Sq0,
	" SQ1", 				&Sq1,
	" 0",					&_0,
	" 1",					&_1,
	" ACC->PM",			&Acc_pm,
	" L^FF",				&L_ff,
	"~L^FF",				&_L_ff,
	" LIR",				&Lir,
	"~LIR",				&_Lir,
	" LDR",				&Ldr,
	"~LDR",				&_Ldr,
	" A0",				&A0,
	"~BRANCH",			&_Branch,
	"~LPA",				&_Lpa,
	" IPC",				&Ipc,
	" A0'",				&A0_,
	"~DELAY",			&_Delay,
	"~LATCH",			&_Latch,
	" READY",			&Ready,
	"~READY",			&_Ready,

	" PAR1",				&Par1,
	" DS1",				&Ds1,
	" DS2",				&Ds2,
	" S0",				&S0,
	" PROG_PAGE",		&Prog_page,
	" ACC",				&Acc,
	" R/~W",				&R_w,
	" SECONDARY",		&Secondary,
	"~SECONDARY",		&_Secondary,
	" INDR",				&Indr,
	" I4_READY",		&I4_ready,
	" LDTIMER (L)",	&Loadtimer,
	" SHIFTED (S)",	&Shifted,
	"~PROCEED (P)",	&_Proceed,
	" XORP    (X)",	&Xorp,
	" XORS    (X)",	&Xors,
	"~RESET   (R)",	&_Reset,
	"~JMP_SEL (J)",	&_Jmpsel,

	"",					NULL,				// place an empty line display

	" C12_12",			&C12_12,
	" C12_6",			&C12_6,
	" C12_8",			&C12_8
};

/*
Print all the signals in the Signal array

Start by printing a status line on the top line of the screen

Status line consists of the following:
   PC registers value,
   The value in the Even Latch
   The value in the Odd Latch
   The value in the Holding Latch
   Whether the 'E'ven or 'O'dd Latch is currently selected
   Value of the Instruction register (T13)
   Value of the Data register (S13)
	Value of the Wait state generator/counter (I14)

   Whether 'W' (memory) 'D' (data reg) or '0' (zero's) is being fed into
   the Acc.

   The current math operation being performed on the Acc.

   The current shift status.
*/

void printSigs( void)
{
	int	ii, sigEntries, rows;
	char	accStr[4];

	// get number of entries in table

	sigEntries = sizeof( Signals) / sizeof( Signals[0]);
	rows = (sigEntries + 2) / 3;

	if (rows < 24)
		rows = 24;

	// print entries in three columns

	fputs( "\x1B[H", stdout);			// ansi.sys - Home cursor

	// print status line

	printf( " PC: %04X [E%02X O%02X H%02X] %c Ir=%02X Dr=%02X I14=%X",
			Ar, Even[Ar >> 1], Odd[Ar >> 1], U9T9, (bit(J14[J14_adr], O5)) ? 'O' : 'E',
			Ir, Dr, I14);

	// get the selected register indicators

	if (!Ds1)
	{
		if (Ds0)
			accStr[0] = 'D';

		else
			accStr[0] = 'W';
	}
	else
		accStr[0] = '0';

	if (!Ds2)
	{
		if (Ds0)
			accStr[1] = 'D';

		else
			accStr[1] = 'W';
	}
	else
		accStr[1] = '0';

	if (!Ds3)
	{
		if (Ds0)
			accStr[2] = 'D';

		else
			accStr[2] = 'W';
	}
	else
		accStr[2] = '0';

	accStr[3] = '\0';

	// print out the current arithmetic operation

	if (!X_arith && !Y)
		printf( " Acc=Acc+%s", accStr);

	else if (X_arith && !Y)
		printf( " Acc=%s    ", accStr);

	else if (!X_arith && Y)
		printf( " Acc=Acc-%s", accStr);

	else if (X_arith && Y)
		printf( " Acc=Acc&%s", accStr);

	// print ACC shift status

	if (!S1 && !S0)
		fputs( " Shf=HOLD ", stdout);

	else if (!S1 && S0)
		fputs( " Shf=LEFT ", stdout);

	else if (S1 && !S0)
		fputs( " Shf=RIGHT", stdout);

	else if (S1 && S0)
		fputs( " Shf=LOAD ", stdout);

	for (ii = 0; ii < rows; ii++)
	{
		printf( "\n%-12s", Signals[ii].name);

		if (Signals[ii].value != NULL)
			printf( " %3u", *Signals[ii].value);

		else
			fputs( "    ", stdout);

		// column 2

		if ((ii + rows) < sigEntries)
		{
			if (Signals[ii].name[0] == '\0')
				putchar( ' ');

			else
				putchar( ',');

			printf( "          %-12s", Signals[ii + rows].name);

			if (Signals[ii + rows].value != NULL)
				printf( " %3u", *Signals[ii + rows].value);

			else
				fputs( "    ", stdout);
		}

		// column 3

		if ((ii + (rows * 2)) < sigEntries)
		{
			if (Signals[ii + rows].name[0] == '\0')
				putchar( ' ');

			else
				putchar( ',');

			printf( "          %-12s", Signals[ii + (rows * 2)].name);

			if (Signals[ii + (rows * 2)].value != NULL)
				printf( " %3u", *Signals[ii + (rows * 2)].value);

			else
				fputs( "    ", stdout);
		}
	}
}

/*
Handle JK Flip Flops

Called with:
   ff = Current value of byte the simulates flip flop
   jj = Current state of J
   kk = Current state of K
   qq = Bit number, in byte, used as Q
  _qq = Bit number, in byte, used as ~Q

Returns:
   New byte value of flip/flop (ff)
*/

uchar	jkff( uchar ff, uchar jj, uchar kk, uchar qq, uchar _qq)
{
	if (jj && !kk)
		ff = (ff | (0x01 << qq)) & ~(0x01 << _qq);

	else if (!jj && kk)
		ff = (ff | (0x01 << _qq)) & ~(0x01 << qq);

	else if (jj && kk)
		ff = ff ^ ((0x01 << qq) | (0x01 << _qq));

	return (ff);
}

// Initialize everything to default states

void setupState( void)
{
	int	ii;

	// initialize the displayed signals to *undefined*
	// if a 0xFF ever pops up we know we forgot to initialize something...

	for (ii = 0; ii < sizeof( Signals) / sizeof( Signals[0]); ii++)
		if (Signals[ii].value != NULL)
			*Signals[ii].value = 0xFF;

	// initialize default values

	Ar = 2;							// address latch
	Ir = Even[0];					// instruction latch
	Dr = Even[0];					// data latch
   U9T9 = Odd[0];					// holding register

	E10 = 0x0C;										// load E10 counter
	B8 = 0;
	A8 = 0;
	B8 = jkff( B8, 0, 1, Q3, _Q2);			// initialize B8 ff's
	B8 = jkff( B8, 0, 1, Q5, _Q6);
	A8 = jkff( A8, 0, 1, Q3, _Q2);			// initialize A8 ff
	I14 = 0x0A;										// load I14 counter

	// user controlled lines

	Loadtimer = 0;					// no timers for now
	Shifted = 0;					// not shfited for now
	_Proceed = 1;					// do not yet proceed
	Xors = 1;						// feed back shift instructions?
	Xorp = 1;
	_Reset = 1;						// reset line
	_Jmpsel = 1;					// Jump selected

	// The 'J register is fixed at address 'JREG'

	Jr = JREG;						// address to jump to if 'jmp' executed
}

// Trickle down all static signals from the source (ultimately the Ir
// register and state counter), down to the last signal being viewed.

void adjStatic( void)
{
	uchar	i10_11;

	// get SQn and Sn from E10

	Sq0 = bit( E10, QA);
	Sq1 = bit( E10, QB);
	_0 = (!Sq0 && !Sq1);
	_1 = Sq0;

	// Prom_adr = IC G6 pin 6 with G14 inverted A0-A3

	if ((Ir & 0xE0) == 0xE0)
		Prom_adr = ~Ir & 0x0F;

	else
		Prom_adr = ((~Ir >> 4) & 0x0F) | 0x10;

	// Decode instruction proms

	Ld_ps = bit( E14[Prom_adr], O1);
	Ld_im = bit( E14[Prom_adr], O2);
	_Vector = bit( E14[Prom_adr], O3);
	_NVector = bit( E14[Prom_adr], O4);
	_Par2 = bit( E14[Prom_adr], O5);
	_ColdStart = bit( E14[Prom_adr], O6);
	S_write_ws = bit( E14[Prom_adr], O7);
	Write_ext = bit( E14[Prom_adr], O8);
	S1 = bit( D14[Prom_adr], O1);
	S0_ = bit( D14[Prom_adr], O2);
	X_arith = bit( D14[Prom_adr], O3);
	Sr24 = bit( D14[Prom_adr], O4);
	Y = bit( D14[Prom_adr], O5);
	Ds0 = bit( D14[Prom_adr], O6);
	Ds1_ = bit( D14[Prom_adr], O7);
	_Wsa_ = bit( D14[Prom_adr], O8);
	Read = bit( C14[Prom_adr], O1);
	Ds3 = bit( C14[Prom_adr], O2);
	Page = bit( C14[Prom_adr], O3);
	_Frame = bit( C14[Prom_adr], O4);
	_Jump = bit( C14[Prom_adr], O5);
	Look_up = bit( C14[Prom_adr], O6);
	Multiply = bit( C14[Prom_adr], O7);
	_Indr_adr = bit( C14[Prom_adr], O8);

	// Check for jumps

	if ((Ir & 0x07) == 0x07)
		_Lpa = 1;

	else if ((Ir & 0x07) == 0x00)
		_Lpa = !(!Sq0 && !_Jump);

	else
		_Lpa = !(!_Jmpsel && !Sq0 && !_Jump);		// use user selected jump

	// get Acc_pm, L_ff, _L_ff from B8

	Acc_pm = _0 && Look_up;
	L_ff = bit( B8, Q5);
	_L_ff = bit( B8, _Q6);

	// get Ready, _Ready, _Branch, _Delay, A0, A0_, _Latch

	_Branch = bit( I14, QD);
	_Delay = bit( I14, QC);
	i10_11 = !(_L_ff && _Branch);
	A0 = bit( Ar, 0);
	A0_ = bit( Ar, 0);
	_Latch = (!i10_11) && (!A0_);
	_Ready = !(_Latch || _Delay);
	Ready = !_Ready;

	// Add _Ready and SQn lines

	Prom_adr |= Sq0 ? 0x20 : 0x00;
	Prom_adr |= Sq1 ? 0x40 : 0x00;
	Prom_adr |= _Ready ? 0x80 : 0x00;

	// translate through F14 prom, set Adv, _Acc_, _Lir_, _Ldr_

	Adv = bit( F14[Prom_adr], O1);
	_Acc_ = bit( F14[Prom_adr], O2);
	_Lir_ = bit( F14[Prom_adr], O3);
	_Ldr_ = bit( F14[Prom_adr], O4);

	// get PROG_PAGE, I4_ready

	F12_6 = !(bit( Ir, 0) || bit( Ir, 1) || bit( Ir, 2) || bit( Ir, 3));
	Prog_page = F12_6 && !_Jump;
	I4_ready = !F12_6 && !bit( Ir, 6) && !bit( Ir, 7) && !_Ready;

	// Get Lir, _Lir, Ldr, _Ldr, Indr

	K4_10 = !(Xorp || Xors);
	C12_12 = !K4_10 && !_Ready && !_NVector;
	C12_6 = !_Ready && !_Frame && !_Proceed;
	C12_8 = !(C12_12 || I4_ready || C12_6);
	Lir = !_Reset || !_Lir_ || !C12_8;
	_Lir = !Lir;
	Ldr = !C12_8 || !_Ldr_;
	_Ldr = !Ldr;
	Indr = !_Ldr_ && !_Indr_adr;

	// get address of J14

	J14_adr = A0 ? 0x01 : 0x00;
	J14_adr |= _Branch ? 0x02 : 0x00;
	J14_adr |= Ldr ? 0x04 : 0x00;
	J14_adr |= _Lpa ? 0x08 : 0x00;
	J14_adr |= L_ff ? 0x10 : 0x00;

	Ipc = bit( J14[J14_adr], O4);
	J14_o1 = bit( J14[J14_adr], O1);
	J14_o2 = bit( J14[J14_adr], O2);
	J14_o5 = bit( J14[J14_adr], O5);

	H14_1 = !(Acc_pm || bit( J14[J14_adr], O3));

	// get B10_11

	A10_8 = !(_1 && Ready);
	D12_6 = !_Jump && !bit( Ir, 3) && !A10_8;
	B10_11 = Multiply || D12_6;

	// get ACC

	D10_10 = !(I4_ready || Loadtimer);
	A14_3 = _1 && Multiply;
	B12_8 = !(A14_3 && Shifted && Ready);
	Acc = !D10_10 || ! _Acc_ || !B12_8;

	S0 = A14_3 || S0_;

	// get PAR1, DS1, DS2

	A14_6 = _0 && Ld_im;
	Par1 = Ld_ps || A14_6;
	Ds1 = A14_6 || Ds1_;
	Ds2 = A14_6 || Ds3 || I4_ready;

	// Get SECONDARY, _SECONDARY

	Secondary = bit( A8, Q3);
	_Secondary = bit( A8, _Q2);

	// Get R~W

	R_w = bit( B8, _Q2);
}

// This routine simulates one CPU clock cycle, by adjusting all affected
// counters and flip/flops

void adjClock( void)
{
	// Latch the 'Odd' (phase 02) latch before toggling clocks

	if (!_Latch)
		U9T9 = Odd[Ar >> 1];

	// Clock data into Instruction and Data registers based on inhibit
	// flags

	if (!_Lir)
	{
		if (J14_o5)
			Ir = U9T9;

		else
			Ir = Even[Ar >> 1];
	}

	if (!_Ldr)
	{
		if (J14_o5)
			Dr = U9T9;

		else
			Dr = Even[Ar >> 1];
	}

	// Increment or load program counter

	if (!_Lpa)
		Ar = Jr;

	else if (Ipc)
		Ar = (Ar + 1) & 0x1F;

	// Increment E10 counter

	if (!_Lir)
		E10 = 0x0C;		// load counter

	else if (Adv)
		E10++;			// increment counter

	// Handle B8 and A8 JK flip/flops

	B8 = jkff( B8, Acc_pm, Ldr, Q5, _Q6);
	B8 = jkff( B8, S_write_ws, Lir, Q3, _Q2);
	A8 = jkff( A8, B10_11, Lir, Q3, _Q2);

	// Increment I14 counter

	if (!H14_1)
	{
		// Load counter

		I14 = J14_o1 ? 0x01 : 0x00;
		I14 |= 0x02;
		I14 |= J14_o2 ? 0x08 : 0x00;
	}
	else if (!_Delay)
		I14++;		// increment counter
}

void main( int argc, char *argv[])
{
	int	cc, ii, eoPtr, romAdr;

	fputs( "\nCINESIM - Cinematronics CPU Simulator.", stderr);
	fputs( "\nVersion 1.0   Written by Zonn Moore.\n", stderr);
	fputs( "\nReleased to the public domain by the author, distribute freely.\n", stderr);

	if (argc < 2)
	{	fputs( "\nUse: CINESIM opc0 [opc1] ... [opc31]\n", stderr);
		fputs( "\nWhere:", stderr);
		fputs( "\n   'opc0' through 'opc31' are opcodes to be simulated.\n", stderr);
		exit( 0);
	}

	eoPtr = 0;			// point to even ROM
	romAdr = 0;

	if (argc > (sizeof( Even) * 2))
		argc = sizeof( Even) * 2;

	for (ii = 0; ii < sizeof( Even); ii++)
	{	Even[ii] = 0;
		Odd[ii] = 0;
	}

	for (ii = 1; ii < argc; ii++)
	{
		if (eoPtr == 0)
		{	sscanf( argv[ii], "%x", &Even[romAdr]);
			eoPtr = 1;
		}
		else
		{	sscanf( argv[ii], "%x", &Odd[romAdr]);
			eoPtr = 0;
			romAdr++;
		}
	}

	fputs( "\x1B[J", stdout);	// ansi.sys - Clear display

	readProms();
	setupState();					// initialize counters and flip/flops
	adjStatic();					// adjust all non-clocked values

	while (1)
	{
		printSigs();

		cc = getch();
		cc = toupper( cc);

		// toggle user defined variables if requested

		if (cc == 'L')
			Loadtimer = !Loadtimer;

		else if (cc == 'S')
			Shifted = !Shifted;

		else if (cc == 'P')
			_Proceed = !_Proceed;

		else if (cc == 'X')
		{	Xorp = !Xorp;
			Xors = !Xors;
		}
		else if (cc == 'R')
			_Reset = !_Reset;

		else if (cc == 'J')
			_Jmpsel = !_Jmpsel;

		else if (cc == 0x1B)
			exit( 0);

		else if (cc == ' ')
			adjClock();			// increment the clock

		adjStatic();			// adjust all non-clocked values
	}
}
