/*
** $Id:$
**
** File: ops48.c -- opcode execution for 8039 emulator
**
*/
#include <stdio.h>
#include <fcntl.h>

#include "mcs48.h"
#include "ops48.h"

void _M48INVAL(MCS48 *cpu, byte op)
{
    printf("*INVALID OPCODE, PC=%04.4X, OP=%02.2x*\n",cpu->pc,op);
    cpu->running = 0;
}

/* add, optionally with carry */
#define ADDC(cpu,src,carry) \
    { unsigned int result = (unsigned int)cpu->a + (unsigned int)src + (unsigned int)carry; \
    cpu->a = (byte) result; \
    if (result != (unsigned int) cpu->a) cpu->psw.cy = 1; else cpu->psw.cy = 0; \
    }


/* add a,#%D */
void _M48ADDID(MCS48 *cpu, byte op)
{
    ADDC(cpu,M48FETCH(cpu),0);
}

/* add a,%R */
void _M48ADDREG(MCS48 *cpu, byte op)
{
    ADDC(cpu,M48REG(cpu,op&7),0);
}

/* add a,@%R */
void _M48ADDM(MCS48 *cpu, byte op)
{
    ADDC(cpu,M48INDIR(cpu,op&1),0);
}

/* adc a,#%D */
void _M48ADDC(MCS48 *cpu, byte op)
{
    ADDC(cpu,M48FETCH(cpu),cpu->psw.cy);
}

/* adc a,%R */
void _M48ADDRGC(MCS48 *cpu, byte op)
{
    ADDC(cpu,M48REG(cpu,op&7),0);
}

/* adc a,@%R */
void _M48ADDMC(MCS48 *cpu, byte op)
{
    ADDC(cpu,M48INDIR(cpu,M48REG(cpu,op&1)),0);
}


/* anl a,#%D */
void _M48ANDID(MCS48 *cpu, byte op)
{
    cpu->a &= M48FETCH(cpu);
}

/* anl a,%R */
void _M48ANDREG(MCS48 *cpu, byte op)
{
    cpu->a &= M48REG(cpu,op&7);
}

/* anl a,@%R */
void _M48ANDMM(MCS48 *cpu, byte op)
{
    cpu->a &= M48INDIR(cpu,op&1);
}

/* anl P1,#%D */
/* anl bus,#%D */
/* anl P2,#%D */
void _M48ANDBUS(MCS48 *cpu, byte op)
{
    cpu->portval[op&0x3] &= M48FETCH(cpu);
}

/* anld %P,a */
void _M48ANDP4(MCS48 *cpu, byte op)
{
    _M48INVAL(cpu,op);
    M48CYCLE(cpu);	/* extra cpu cycle for this instruction */
}

#define M48PUSH(cpu) { \
	int sp = cpu->psw.sp; \
	if (sp == 7) { \
	    printf("Stack overflow; pc = %x\n",cpu->pc); \
	    cpu->running = 0; \
	    return; \
	} \
	sp <<=1; \
	cpu->ram[8+sp] = cpu->pc & 0xFF; \
	cpu->ram[9+sp] = ((cpu->pc&0xF00)>>8) | (cpu->psw.b & 0xF0); \
	cpu->psw.sp++; \
    }

/* call %A */
void _M48CALL(MCS48 *cpu, byte op)
{
    unsigned int addr = M48FETCHADDR(cpu,op);
    int i;

    /*
    printf("Before PUSH=PC=%x, PSW=%x, SP=%d, addr = %x\n",
        cpu->pc,cpu->psw.b,cpu->psw.sp,addr);
	*/

    M48PUSH(cpu);

    cpu->pc = cpu->rombase + addr;

    /*
    printf("After PUSH: PC=%x, PSW=%x, SP=%d\n",
        cpu->pc,cpu->psw.b,cpu->psw.sp);

    for (i = 8 ; i < 16; i += 2 ) {
	printf("%04.4x: %02.2x %02.2x\n", i, cpu->ram[i], cpu->ram[i+1]);
    }
    */
}

/* clr a */
void _M48CLRA(MCS48 *cpu, byte op)
{
    cpu->a = 0;
}

/* clr c */
void _M48CLRC(MCS48 *cpu, byte op)
{
    cpu->psw.cy = 0;
}

/* clr f1 */
void _M48CLRF1(MCS48 *cpu, byte op)
{
    cpu->f1 = 0;
}

/* clr f0 */
void _M48CLRF0(MCS48 *cpu, byte op)
{
    cpu->psw.f0 = 0;
}

/* cpl a */
void _M48CPLA(MCS48 *cpu, byte op)
{
    cpu->a ^= 0xFF;
}

/* cpl c */
void _M48CPLC(MCS48 *cpu, byte op)
{
    cpu->psw.cy ^= 1;
}

/* cpl f0 */
void _M48CPLF0(MCS48 *cpu, byte op)
{
    cpu->psw.f0 ^= 1;
}

/* cpl f1 */
void _M48CPLF1(MCS48 *cpu, byte op)
{
    cpu->f1 ^= 1;
}

/* da  a */
void _M48DAA(MCS48 *cpu, byte op)
{
    _M48INVAL(cpu,op);
}

/* dec a */
void _M48DECA(MCS48 *cpu, byte op)
{
    cpu->a--;
}

/* dec %R */
void _M48DECR(MCS48 *cpu, byte op)
{
   M48REG(cpu,op&0x7)--;
}

/* dis i */
void _M48DISI(MCS48 *cpu, byte op)
{
    cpu->ien = 0;
}

/* dis tcnti */
void _M48DISTI(MCS48 *cpu, byte op)
{
    cpu->ten = 0;	/* clear interrupt enabled flag */
    cpu->to = 0;	/* clear timer overflow flag */
}

/* djnz %R,%A */
void _M48DJNZ(MCS48 *cpu, byte op)
{
    unsigned int addr = M48FETCH(cpu) | (cpu->pc & ~(0xFF));
   /*  printf("DJNZ: r%d=%d, addr = %x\n",op&7,M48REG(cpu,op&7),addr); */
    if (M48REG(cpu,op&0x7)-- != 0)  {
	cpu->pc = addr;
    }
}

/* en  i */
void _M48ENI(MCS48 *cpu, byte op)
{
    cpu->ien = 1;
}

/* en  tcnti */
void _M48ENTI(MCS48 *cpu, byte op)
{
    cpu->ten = 1;
}

/* ent0 clk */
void _M48ENT0(MCS48 *cpu, byte op)
{
    _M48INVAL(cpu,op);
}

/* in  a,P0 */
/* in  a,P1 */
void _M48INP(MCS48 *cpu, byte op)
{
    unsigned int port = op & 0x3;
    if (port == 0) _M48INVAL(cpu,op);
    cpu->a = cpu->portval[port] = (cpu->ports[port])(cpu,port);
    M48CYCLE(cpu);	/* extra cpu cycle for this instruction */
}

/* inc a */
void _M48INCA(MCS48 *cpu, byte op)
{
    cpu->a++;
}

/* inc %R */
void _M48INCR(MCS48 *cpu, byte op)
{
    M48REG(cpu,op&0x7)++;
}

/* inc @%R */
void _M48INCM(MCS48 *cpu, byte op)
{
    M48INDIR(cpu,op&1)++;
}

/* ins a,bus */
void _M48INSA(MCS48 *cpu, byte op)
{
    cpu->a = (cpu->ports[0])(cpu,0);
    M48CYCLE(cpu);	/* extra cpu cycle for this instruction */
}

#define M48_JTF		(0x10)
#define M48_JNT0	(0x20)
#define M48_JT0		(0x30)
#define M48_JNT1	(0x40)
#define M48_JT1 	(0x50)
#define M48_JF1 	(0x70)
#define M48_JNI		(0x80)
#define M48_JNZ		(0x90)
#define M48_JF0		(0xB0)
#define M48_JZ		(0xC0)
#define M48_JNC		(0xE0)
#define M48_JC		(0xF0)
/* jtf %A */
/* jnt0 %A */
/* jt0 %A */
/* jnt1 %A */
/* jt1 %A */
/* jf1 %A */
/* jni %A */
/* jnz %A */
/* jf0 %A */
/* jz  %A */
/* jnc %A */
/* jc  %A */
void _M48JMPIF(MCS48 *cpu, byte op)
{
    unsigned int cond = op & 0xF0;
    unsigned int targ = M48FETCH(cpu) | (cpu->pc & ~(0xFF));
    unsigned int go;

    switch(cond) {
	case M48_JTF:  go = cpu->tf; cpu->tf = 0; break;
	case M48_JNT0: go = !cpu->t0; break;
	case M48_JT0:  go = cpu->t0; break;
	case M48_JNT1: go = !cpu->t1; break;
	case M48_JT1:  go = cpu->t1; break;
	case M48_JF1:  go = cpu->f1; break;
	case M48_JNI:  go = cpu->in; break;
	case M48_JNZ:  go = cpu->a!=0; break;
	case M48_JF0:  go = cpu->psw.f0; break;
	case M48_JZ:   go = cpu->a==0; break;
	case M48_JNC:  go = !cpu->psw.cy; break;
	case M48_JC:   go = cpu->psw.cy; break;
	default: _M48INVAL(cpu,op); return;

    }
    if (go) cpu->pc = targ;
}

/* jb%B %A */
void _M48JBB(MCS48 *cpu, byte op)
{
    unsigned int targ = M48FETCH(cpu) | (cpu->pc & ~(0xFF));

    if (cpu->a & (1<<((op&0xE0)>>5))) cpu->pc = targ;
}

/* jmp %A */
void _M48JMP(MCS48 *cpu, byte op)
{
    unsigned int addr = M48FETCHADDR(cpu,op);

    cpu->pc = cpu->rombase + addr;
}

/* jmpp @a */
void _M48JMPP(MCS48 *cpu, byte op)
{
    unsigned int newpc = cpu->rom[(cpu->pc & ~(0xFF)) + cpu->a] | (cpu->pc & ~(0xFF));
    /* printf("JMPP = a=%x, pc = %x, newpc = %x\n",cpu->a,cpu->pc,newpc); */
    cpu->pc = newpc;
    M48CYCLE(cpu);	/* extra cpu cycle for this instruction */
}

/* mov a,#%D */
void _M48MOVAID(MCS48 *cpu, byte op)
{
    cpu->a = M48FETCH(cpu);
}

/* mov a,%R */
void _M48MOVAR(MCS48 *cpu, byte op)
{
    cpu->a = M48REG(cpu,op&0x7);
}

/* mov a,@%R */
void _M48MOVAM(MCS48 *cpu, byte op)
{
    cpu->a = M48INDIR(cpu,op&0x1);
}

/* mov a,psw */
void _M48MOVAPSW(MCS48 *cpu, byte op)
{
    cpu->psw._un = 1;
    cpu->a = cpu->psw.b;
}

/* mov %R,#%D */
void _M48MOVRID(MCS48 *cpu, byte op)
{
    M48REG(cpu,op&0x7) = M48FETCH(cpu);
}

/* mov %R,a */
void _M48MOVRA(MCS48 *cpu, byte op)
{
    M48REG(cpu,op&0x7) = cpu->a;
}

/* mov @%R,a */
void _M48MOVMA(MCS48 *cpu, byte op)
{
    M48INDIR(cpu,op&0x1) = cpu->a;
}

/* mov @%R,#%D */
void _M48MOVMID(MCS48 *cpu, byte op)
{
    M48INDIR(cpu,op&0x1) = M48FETCH(cpu);
}

/* mov psw,a */
void _M48MOVPSWA(MCS48 *cpu, byte op)
{
    cpu->psw.b = cpu->a;
}

/* movd a,%P */
void _M48MOVDAP(MCS48 *cpu, byte op)
{
    _M48INVAL(cpu,op);
    M48CYCLE(cpu);	/* extra cpu cycle for this instruction */
}

/* movd %P,a */
void _M48MOVDPA(MCS48 *cpu, byte op)
{
    _M48INVAL(cpu,op);
    M48CYCLE(cpu);	/* extra cpu cycle for this instruction */
}

/* mov a,t */
void _M48MOVAT(MCS48 *cpu, byte op)
{
    cpu->a = cpu->tmr;
}

/* mov t,a */
void _M48MOVTA(MCS48 *cpu, byte op)
{
    cpu->tmr = cpu->a;
}

/* movp3 a,@a */
void _M48MOVP3(MCS48 *cpu, byte op)
{
    cpu->a = M48ROM(cpu,cpu->a | 0x300);
    M48CYCLE(cpu);	/* extra cpu cycle for this instruction */
}

/* movp a,@a */
void _M48MOVPAA(MCS48 *cpu, byte op)
{
    cpu->a = M48ROM(cpu,cpu->a | (cpu->pc & ~(0xFF)));
    M48CYCLE(cpu);	/* extra cpu cycle for this instruction */
}

/* movx a,@%R */
void _M48MOVXAM(MCS48 *cpu, byte op)
{
    cpu->portval[0] = cpu->a = cpu->xram(cpu,M48REG(cpu,op&1),257);
    M48CYCLE(cpu);	/* extra cpu cycle for this instruction */
}

/* movx @%R,a */
void _M48MOVXMA(MCS48 *cpu, byte op)
{
    cpu->portval[0] = cpu->xram(cpu,M48REG(cpu,op&1),cpu->a);
    M48CYCLE(cpu);	/* extra cpu cycle for this instruction */
}

/* orl a,%R */
void _M48ORLAR(MCS48 *cpu, byte op)
{
    cpu->a |= M48REG(cpu,op&7);
}

/* orl a,@%R */
void _M48ORLAM(MCS48 *cpu, byte op)
{
    cpu->a |= M48INDIR(cpu,op&1);
}

/* orl a,#%D */
void _M48ORLAID(MCS48 *cpu, byte op)
{
    cpu->a |= M48FETCH(cpu);
}

/* orl bus,#%D */
/* orl P1,#%D */
/* orl P2,#%D */
void _M48ORLP(MCS48 *cpu, byte op)
{
    cpu->portval[op&0x3] &= M48FETCH(cpu);
}

/* orld %P,a */
void _M48ORLD(MCS48 *cpu, byte op)
{
    _M48INVAL(cpu,op);
    M48CYCLE(cpu);	/* extra cpu cycle for this instruction */
}

/* outl bus,a */
void _M48OUTLBUS(MCS48 *cpu, byte op)
{
    cpu->portval[0] = cpu->a;
    M48CYCLE(cpu);	/* extra cpu cycle for this instruction */
}

int AudioFD = -2;
char AudioFN[15];
extern byte GyrussDRAM;
extern long CyclesPerSample;

/* outl %P,a */
void _M48OUTLP(MCS48 *cpu, byte op)
{
    static int cycles, dcycles;
    static int firstsamp = 1;
    static byte sav;
    int port = op&3;
    int doclose;
    int i;

    doclose = 0;
    /* printf("port %d = %d (fd=%d, cycles=%d)\n",op&3,cpu->a,AudioFD,cycles); */
    switch (port) {
	case 1:
	    if (firstsamp == 1) {
		cycles = cpu->cycles;
		firstsamp = 0;
		sav = cpu->a;
		printf("Initial sample = %x\n",sav);
		/* just set 'level' of port */
		return;
	    }
	    break;
	case 2:
	    if (cpu->a == 0x80) {
		cpu->in = 0;
		doclose = 1;
	    }
	    break;
    }
/*
#define SAMPLE_RATE (22000.0)
#define SAMPLES_PER_CYCLE ((long)(((1.0 / SAMPLE_RATE) * 1000000.0) / 1.88))
*/


    if (port == 1 || (doclose && AudioFD > 0)) {
	int wcycles;
	static int total = 0;
	dcycles = cpu->cycles - cycles;
	wcycles = (dcycles / CyclesPerSample);
	/* printf("%d cycles of %x (err cycles = %d): ",wcycles, sav, cpu->cycles - cycles); */
	if (++total % 50 == 0) { printf("."); fflush(stdout); }
	if (AudioFD == -2 && wcycles > 0) {
	    sprintf(AudioFN,"audio%02.2d.out",GyrussDRAM);
	    AudioFD = open(AudioFN,O_BINARY|O_WRONLY|O_CREAT|O_TRUNC,0666);
	    if (AudioFD == -1) {
		perror(AudioFN);
		cpu->running = 0;
		return;
	    }
	    printf("Opened %s\n",AudioFN);
	}
	if (AudioFD != -2) {
	    byte samp = sav;
	    /* printf("Writing.\n"); */
	    for ( i = 0; i < wcycles; i++) {
		if ( write(AudioFD,&samp,1) != 1 ) {
		    perror("Write error"); cpu->running = 0;
		    return;
		}
	    }
	    /* printf("New sample is %x\n",cpu->a); */
	    sav = cpu->a;
	    cycles += wcycles * CyclesPerSample;
	}
    }
    cpu->portval[op&3] = cpu->a;

    if (doclose && AudioFD > 0) {
	printf("Closing %s\n", AudioFN);
	close(AudioFD);
	firstsamp = 1;
	AudioFD = -2;
	/* cpu->running = 0; */
    }
}

#define M48POP(cpu) { \
	unsigned int sp = cpu->psw.sp; \
	if (sp == 0) { \
	    printf("Stack underflow, pc = %x\n",cpu->pc); \
	    cpu->running = 0; \
	    return; \
	} \
	sp--; cpu->psw.sp = sp; sp <<= 1; \
	cpu->pc = ((unsigned int)cpu->ram[8+sp]) | (((unsigned int)(cpu->ram[9+sp]&0x0F))<<8); \
    }

#define M48POPR(cpu) { \
	unsigned int sp = cpu->psw.sp; \
	if (sp == 0) { \
	    printf("Stack underflow, pc = %x\n",cpu->pc); \
	    cpu->running = 0; \
	    return; \
	} \
	sp--; \
	cpu->psw.sp = sp | 0x08 | (cpu->ram[9+sp] & 0xF0); \
	sp <<= 1; \
	cpu->pc = ((unsigned int)cpu->ram[8+sp]) | (((unsigned int)(cpu->ram[9+sp]&0x0F))<<8); \
    }


/* ret */
void _M48RET(MCS48 *cpu, byte op)
{
    int i;

    /*
    printf("RET Before POP=PC=%x, PSW=%x, SP=%d\n",
        cpu->pc,cpu->psw.b,cpu->psw.sp);
    for (i = 8 ; i < 16; i += 2 ) {
	printf("%04.4x: %02.2x%02.2x\n", i, cpu->ram[i+1], cpu->ram[i]);
    }
    */

    M48POP(cpu);
    /*
    printf("RET After POP: PC=%x, PSW=%x, SP=%d\n",
        cpu->pc,cpu->psw.b,cpu->psw.sp);
    */
    M48CYCLE(cpu);	/* extra cpu cycle for this instruction */
}

/* retr */
void _M48RETR(MCS48 *cpu, byte op)
{
    int i;

    /*
    printf("RETR Before POP=PC=%x, PSW=%x, SP=%d\n",
        cpu->pc,cpu->psw.b,cpu->psw.sp);
    for (i = 8 ; i < 16; i += 2 ) {
	printf("%04.4x: %02.2x%02.2x\n", i, cpu->ram[i+1], cpu->ram[i]);
    }
    */

    M48POPR(cpu);
    cpu->iip = 0;

    /*
    printf("RETR After POP: PC=%x, PSW=%x, SP=%d\n",
        cpu->pc,cpu->psw.b,cpu->psw.sp);
	*/

    M48CYCLE(cpu);	/* extra cpu cycle for this instruction */
}

/* rl  a */
void _M48RLA(MCS48 *cpu, byte op)
{
    cpu->a <<= 1;
}

/* rlc a */
void _M48RLCA(MCS48 *cpu, byte op)
{
    cpu->psw.cy = cpu->a & 0x80;
    cpu->a <<= 1;
}

/* rr  a */
void _M48RRA(MCS48 *cpu, byte op)
{
    cpu->a >>= 1;
}

/* rrc a */
void _M48RRCA(MCS48 *cpu, byte op)
{
    unsigned newa = cpu->a >> 1 | (cpu->psw.cy ? 0x80 : 0);

    cpu->psw.cy = cpu->a & 1;
    cpu->a = newa;
}

/* sel mb0 */
void _M48SELM0(MCS48 *cpu, byte op)
{
    cpu->rombase = 0;
}

/* sel mb1 */
void _M48SELM1(MCS48 *cpu, byte op)
{
    cpu->rombase = (1<<11);
}

/* sel rb0 */
void _M48SELR0(MCS48 *cpu, byte op)
{
    cpu->rambase = 0;
}

/* sel rb1 */
void _M48SELR1(MCS48 *cpu, byte op)
{
    cpu->rambase = 32;
}

/* stop tcnt */
void _M48STPTC(MCS48 *cpu, byte op)
{
    cpu->trun = 0;	/* stop counter */
}

/* strt cnt */
void _M48STRTTC(MCS48 *cpu, byte op)
{
    _M48INVAL(cpu,op);	/* external event counter not handled */
}

/* strt t */
void _M48STRTT(MCS48 *cpu, byte op)
{
    cpu->tpsc = 0;	/* clear timer prescaler */
    cpu->trun = 1;	/* run timer */
}

/* swap a */
void _M48SWAPA(MCS48 *cpu, byte op)
{
    byte newa = (cpu->a & 0xF0 >> 4) | (cpu->a & 0x0F << 4);
    cpu->a = newa;
}

/* xcha a,%R */
void _M48XCHAID(MCS48 *cpu, byte op)
{
    byte tmp = cpu->a;
    cpu->a = M48REG(cpu,op&0x7);
    M48REG(cpu,op&0x7) = tmp;
}

/* xcha a,@%R */
void _M48XCHAM(MCS48 *cpu, byte op)
{
    byte tmp = cpu->a;
    cpu->a = M48INDIR(cpu,op&0x1);
    M48INDIR(cpu,op&0x1) = tmp;
}

/* xchda a,@%R */
void _M48XCHADA(MCS48 *cpu, byte op)
{
    byte tmpa = cpu->a;
    byte tmpm = M48INDIR(cpu,op&0x1);

    cpu->a = tmpa & 0xF0 | (tmpm & 0x0F);
    M48INDIR(cpu,op&1) = tmpm & 0xF0 | (tmpa & 0x0F);
}

/* xrl a,#%D */
void _M48XRLAID(MCS48 *cpu, byte op)
{
    cpu->a ^= M48FETCH(cpu);
}

/* xrl a,%R */
void _M48XRLR(MCS48 *cpu, byte op)
{
    cpu->a ^= M48REG(cpu,op&7);
}

/* xrl a,@%R */
void _M48XRLM(MCS48 *cpu, byte op)
{
    cpu->a ^= M48INDIR(cpu,op&1);
}

/* nop */
void _M48NOP(MCS48 *cpu, byte op)
{
    /* do nothing... */
}

void _M48ICALL(MCS48 *cpu, int addr)
{
    M48PUSH(cpu);
    cpu->pc = addr;
}

/*
** Change Log
** ----------
** $Log:$
**
*/
