/*
** $Id:$
**
** File: mcs48.h -- 8048/8039 emulator
**
** This program is originally part of the GYRUSS emulator; GYRUSS uses an
** 8039 as an FX synthesizer on it's audio PCB.  The GRYUSS emulator runs
** the 8039 through it's paces to generate audio samples which it
** will then play back on cue; this saves us the online emulation of
** yet-another-cpu
**
*/
#ifndef _H_MCS48_
#define _H_MCS48_

typedef unsigned char byte;

#ifndef INLINE
#define INLINE 
#endif

/*
** the MCS-48 architecture is a Harvard Architecture; what this means
** is that I and D space are seperate
*/
#define M48_VEC_RESET (0)	/* code addr for reset vector */
#define M48_VEC_INTR  (3)	/* code addr for external interrupt vector */
#define M48_VEC_TIMER (7)	/* code addr for timer interrupt vector */

typedef struct mcs48_f MCS48;

/* called for a read of a CPU port */
typedef byte (*M48PortHandler)(MCS48 *cpu, int port);
/* called each clock phase (3 times per instruction) */
typedef void (*M48ClkoutHandler)(MCS48 *cpu);
/* typedef of an opcode function */
typedef void (*M48OpcodeFunc)(MCS48 *cpu, byte op);
/* called for read and write of 'external store' */
typedef byte (*M48XRamHandler)(MCS48 *cpu, byte addr, int val);

/* definition of the PSW;  this definition assumes that bitfields
** fill in a byte from LSb to MSb; if your compiler does not
** do this, define MCS48_REVERSE_BITS
*/
typedef union mcs48psw_f {
    struct {
#ifndef MCS48_REVERSE_BITS
	 /* LSb to MSb */
	 unsigned int sp:3;	/* three bits of stack pointer */
	 unsigned int _un:1;	/* '1' */
	 unsigned int bs:1;	/* Bank Select (mirrored in 'rambase') */
	 unsigned int f0:1;	/* flag0 bit */
	 unsigned int ac:1;	/* aux carry */
	 unsigned int cy:1;	/* accumulator carry */
#else
	 /* reversed, MSb to LSb */
	 unsigned int cy:1;	/* accumulator carry */
	 unsigned int ac:1;	/* aux carry */
	 unsigned int f0:1;	/* flag0 bit */
	 unsigned int bs:1;	/* Bank Select (mirrored in 'rambase') */
	 unsigned int _un:1;	/* '1' */
	 unsigned int sp:3;	/* three bits of stack pointer */
#endif
    };
    byte b;	/* the whole thing */
} MCS48PSW;
#define M48_DEFAULT_PSW (0x08)

/* main CPU structure */
struct mcs48_f {
    byte *rom;		/* program store */
    byte *ram;		/* system ram */
    int type;		/* cpu type */
    byte a;		/* accumulator */
    MCS48PSW psw;	/* current program status word */
    short pc;		/* program counter */
    int cputype;	/* type of cpu */
#define MCS48_8039 (0)
    int regmask;	/* mask of valid bits in register for indirect */
    int running;	/* is cpu running? */
    int rambase;	/* which RAM bank are we in (0 or 24) */
    int rombase;	/* offset to PC for all fetches */
    int highbank;	/* what the 'high' offset is to the PC */
    int ramsize, romsize;
    unsigned long cycles;	/* cycle count since reset */
    M48PortHandler ports[3];	/* I/O ports (BUS,P1,P2) */
    M48XRamHandler xram;	/* read 'external' store */
    int ien;			/* interrupt enabled */
    int iip;			/* interrupt in progress? */
    int t0, t1, in;		/* 'external' inputs */
    int f1;			/* flag1 */

    int tf, to;			/* timer flag, indicates overflow */
    byte tmr;			/* current timer value */
    int ten;			/* timer interrupt enabled */
    int trun;			/* timer running */
    int tpsc;			/* timer prescaler */

    byte portval[3];		/* latched value of I/O ports */
};

/* get a byte from ROM */
#define M48ROM(cpu,addr) ((cpu)->rom[(addr)])

#define M48_PCMASK (0x7FF)
/* increment PC; wrap at end of 'bank' */
#define M48INCPC(cpu) (((cpu)->pc)++, ((cpu)->pc) &= M48_PCMASK)

/* fetch next byte of program store, increment PC and cycle count */
#define M48FETCH(cpu) ( M48INCPC(cpu), M48CYCLE(cpu), M48ROM(cpu,cpu->pc-1))
/*
INLINE byte M48FETCH(MCS48 *cpu) {
     M48INCPC(cpu);
     M48CYCLE(cpu);
     return M48ROM(cpu,cpu->pc-1);
}
*/

/* fetch register */
#define M48REG(cpu,reg) ((cpu)->ram[(reg)+((cpu)->rambase)])

/* register indirect addressing */
#define M48INDIR(cpu,reg) ((cpu)->ram[((M48REG((cpu),reg)&((cpu)->regmask)))])

/* return '11-bit' address from current opcode and next byte */
#define M48FETCHADDR(cpu,op) \
    (((int)((op) & 0xE0)) << 3 | M48FETCH(cpu))


/*
** interrupt timer handling by emulator:
**
** six flags are kept track of:
**
** tf - timer flag, indicates that an overflow has happend. cleared by
**      reset or 'JTF' instruction
**
** to - timer overflow. indicates that an overflow has happened.  reset
**	by timer interrupt recognized, chipreset or timer int disabled.
**      if timer int is disabled, this flag will never get set.
**
** ten - timer interrupt enable. set by "EN TCNT1" instruction, cleared
**	by "DIS TCNT1" instruction and chip reset
**
** int - interrupt pin.  set or cleared by external routines indicating
**	hardware interrupt.  only sampled after last cycle of instruction
**
** ien - interrupts enabled.  set by "EN I" instruction, cleared by reset
**	and "DIS I" instruction.
**
** iip - interrupt in progress.  set by this formula: (int & ien | to).
**	cleared by chip reset and "RETR" instruction
**
** interrupts are only dispatched just before the fetch of a new instruction
** by calling the function '_INTCALL(cpu,int)' where int=3 for 'in' and '7'
** for 'to'.
*/

/*
** update timer for 1 machine cycle
** a) increment prescaler
** a) if the timer is running && prescaler >= 32
*/
INLINE M48CYCLE(MCS48 *cpu)
{
    /* increment timer prescaler */
    cpu->tpsc++; 						
    cpu->cycles++; /* increment total-cycle count */
    /* if timer is running, and prescaler overflows */
    if (cpu->trun && cpu->tpsc > 0x1F) {
	/* increment timer */
	cpu->tmr++;
	if (cpu->tmr == 0) { /* timer just overflowed */	
	    if (cpu->ten) cpu->to = 1;     /* set timer overflow */
	    cpu->tf = 1;     /* set timer flag */
	}
	cpu->tpsc &= 0x1F;
    }
    /* printf("[TP=%x]",cpu->tpsc); */
}

int M48Init(MCS48 *cpu, int ramsize, int romsize);
void M48Reset(MCS48 *cpu);
/* execute 1 instruction */
int M48Run(MCS48 *cpu);

/* read a rom file (binary format) */
int M48LoadRom(MCS48 *cpu,char *file, int org, int size);
/* save CPU state to a file */
int M48SaveCPU(MCS48 *cpu, char *file);
/* load CPU state from a file */
int M48LoadCPU(MCS48 *cpu, char *file);

/* generate CPU jump table for instructions; call after ParseOps */
int M48GenFunc();

#endif /* _H_MCS48_ */

/*
** Change Log
** ----------
** $Log:$
**
*/
