/*
** $Id:$
**
** File: mcs48.c -- 8048/8039 emulator
**
** This program is originally part of the GYRUSS emulator; GYRUSS uses an
** 8039 as an FX synthesizer on it's audio PCB.  The GRYUSS emulator runs
** the 8039 through it's paces to generate audio samples which it
** will then play back on cue; this saves us the online emulation of
** yet-another-cpu
*/
#include <stdio.h>
#include <fcntl.h>
#include <malloc.h>
#include <sys/stat.h>
#include <signal.h>

#undef PERF_TIME
#define DEBUG

#include "mcs48.h"

#ifdef PERF_TIME
#include <windows.h>
#include <mmsystem.h>
#pragma comment(lib,"winmm.lib")
#endif

#ifndef MIN
#define MIN(a,b) ((a) > (b) ? (a) : (b))
#endif

#define DIS48_INCLUDE
/* Watch out: including a "C" file ... */
#include "dis48.c"

/* handlers for each emulated opcode */
static M48OpcodeFunc __M48[256];

extern void _M48INVAL(MCS48 *cpu, byte op);

/* default port handler, called when CPU reads port */
static byte __M48PortHandler(MCS48 *cpu, int port)
{
    return 0;
}

/*
** this function just returns 0, but a real function would
** simulate what happens when external data at address 'addr' is
** read (val>256) or written to (val < 256)
*/
static byte __M48XRamHandler(MCS48 *cpu, byte addr, int val)
{
    return 0;
}

typedef struct m48cpu_info_f {
    int type;		/* type # */
    char *name;		/* name of cpu */
    int introm;		/* size of internal ROM */
    int ramsize;	/* size of internal RAM */
    int regmask;	/* register-indirect addressing mask */
    int highbank;	/* offset to 'high' bank of ROM */
} M48CpuInfo;

static M48CpuInfo __CpuInfo[] = {
    { MCS48_8039,
      "8039",	/* string name */
      0,	/* internal rom size */
      64,	/* internal RAM size */
      0x3F,	/* lower 6 bits used for register-indirect addressing */
    }
};


int M48Init(MCS48 *cpu, int type, int extromsize)
{
    M48CpuInfo *i = __CpuInfo + type;

    if (__M48[0] == NULL) M48GenFunc();

    cpu->type = type;
    cpu->romsize = i->introm + extromsize,
    cpu->ramsize = i->ramsize;
    cpu->regmask = i->regmask;
    cpu->highbank = i->highbank;

    cpu->rom = (unsigned char *)calloc(cpu->romsize, sizeof(byte));
    cpu->ram = (unsigned char *)calloc(cpu->ramsize, sizeof(byte));

    if (cpu->rom == NULL || cpu->ram == NULL) return -1;

    /*
    ** install default (do-nothing handlers for I/O routines ...
    ** the user will want to override these ...
    */
    cpu->ports[0] = cpu->ports[1] = __M48PortHandler;

    cpu->xram = __M48XRamHandler;

    cpu->a = 0;		/* accumulator starts at 0 */
    cpu->tpsc = 0;	/* reset timer prescaler */

    /* make sure CPU is in reset state */
    M48Reset(cpu);
    return(0);
}

void M48Reset(MCS48 *cpu)
{
    cpu->portval[0] = cpu->portval[2] = 0;

    cpu->psw.b = M48_DEFAULT_PSW;

    cpu->rambase = 0;
    cpu->rombase = 0;	/* reset memory bank */
    cpu->pc = 0;
    cpu->psw.sp = 0;
    cpu->psw.f0 = 0;
    cpu->f1 = 0;

    /* reset interrupt handling */
    cpu->ien = 0;	/* interrupts disabled */
    cpu->in = 0;	/* reset interrupt 'pin' flip */
    cpu->iip = 0;	/* no interrupt in progress */

    /* reset timer */
    cpu->to = 0;	/* no overflow */
    cpu->tmr = 0;
    cpu->tf = 0;
    cpu->ten = 0;	/* timer disabled */

    cpu->cycles = 0;

    /* start out not running ... */
    cpu->running = 0;
}

int M48Run(MCS48 *cpu)
{
    byte op;

    cpu->running = 1;	/* yes, mathilda, we are executing */

    /* check for interrupt ... */
    if (((cpu->in & cpu->ien) | (cpu->to)) && !cpu->iip) {
	printf("Interrupt .. \n");
	cpu->iip = 1;
	_M48ICALL(cpu,cpu->to ? M48_VEC_TIMER : M48_VEC_INTR );
    } else {
	op = M48FETCH(cpu);
	(__M48[op])(cpu,op);
    }

    return (cpu->running);
}

int M48LoadRom(MCS48 *cpu,char *file, int org, int size)
{
    int fd;
    int fsize;
    struct stat sb;

    if ((fd = open(file,O_RDONLY|O_BINARY)) < 0) return (-1);

    if (fstat(fd,&sb) < 0) return (-2);

    fsize = MIN(cpu->romsize, sb.st_size);
    if (size != -1) {
	fsize = MIN(size,fsize);
    }
    printf("Loading system ROM rom '%s' at org $%04.4x, %d bytes (%x)\n",
    	file,org,fsize,fsize);
    if ((size = read(fd,cpu->rom,fsize)) < fsize) {
	if (size == -1) {
	    perror("Read Error:"); return (-3);
	}
	printf("Short read: got %d, wanted %d\n",size,fsize);
    }
    close(fd);
    return(0);	/* all's well that ends well */
}

/* save CPU state to a file */
int M48SaveCPU(MCS48 *cpu, char *file)
{
    printf("SaveCPU not implemented, yet.\n");
    return(-1);
}

/* load CPU state from a file */
int M48LoadCPU(MCS48 *cpu, char *file)
{
    printf("LoadCPU not implemented, yet.\n");
    return(-1);
}

/* generate CPU jump table for instructions */
void M48GenFunc()
{
    int i, op, by;

    printf("Generating mcs-48 CPU\n");
    M48ParseOps();

    for ( by = 0 ; by < 256; by++ ) {
	/* printf("Gen %d\n",by); */
	op = -1;
	for ( i = 0; i < MAX_OPS; i++) {
	    /* printf("search %d (%x)\n",i,Op[i].bits); */
	    if ((by & Op[i].mask) == Op[i].bits) {
		if (op != -1) {
		    printf("Warning: opcode %02.2X matches %d (%s) and %d (%s)\n",
			by,i,Op[i].fmt,op,Op[op].fmt);
		}
		op = i;
	    }
	}
	if (op == -1) {
	    printf("Found invalid opcode %x\n",by);
	} else {
	    __M48[by] = Op[op].func;
	    /* printf("found %d: %s\n",op,Op[op].fmt); */
	}
    }
    by = 0;
    for ( i = 0 ; i < 256; i++) {
	if (__M48[i] == NULL) { by++; __M48[i] = _M48INVAL; }
    }
    printf("M48GenFunc: %d invalids generated\n",by);
}

void M48DumpRegs(MCS48 *cpu)
{
    int i;
    printf("    A:  %3.3d (0x%02.2x)  p0:%3.3d (0x%02.2x)  p1:%3.3d (0x%02.2x)  p2:%3.3d (0x%02.2x)\n",
    	cpu->a,cpu->a,
	cpu->portval[0], cpu->portval[0],
	cpu->portval[1], cpu->portval[1],
	cpu->portval[2], cpu->portval[2]);

    printf("    PC: 0x%4.4x, PSW:%x (BNK:%d F0:%d AC:%d CY:%d SP:%d)\n",
        cpu->pc, cpu->psw.b,
	cpu->psw.bs ? 1 : 0,
	cpu->psw.f0 ? 1 : 0,
	cpu->psw.ac ? 1 : 0,
	cpu->psw.cy ? 1 : 0,
	cpu->psw.sp);

    printf("    ien:%d  iip:%d  f1:%d  \n"
    	   "    tf:%d  to:%d  tmr=%02.2X  ten=%d  trun=%d  tpsc=%d\n",
	   cpu->ien, cpu->iip, cpu->f1, cpu->tf, cpu->to, cpu->tmr, cpu->ten,
	   cpu->trun, cpu->tpsc);
    printf("    R0:%.3d (0x%02.2x)  R1:%.3d (0x%2.2x) R1:%.3d (0x%2.2x) R1:%.3d (0x%2.2x)\n",
    	M48REG(cpu,0), M48REG(cpu,0), M48REG(cpu,1), M48REG(cpu,1),
    	M48REG(cpu,2), M48REG(cpu,2), M48REG(cpu,3), M48REG(cpu,3));
    printf("    R4:%.3d (0x%02.2x)  R5:%.3d (0x%2.2x) R6:%.3d (0x%2.2x) R7:%.3d (0x%2.2x)\n",
    	M48REG(cpu,4), M48REG(cpu,4), M48REG(cpu,5), M48REG(cpu,5),
    	M48REG(cpu,6), M48REG(cpu,6), M48REG(cpu,7), M48REG(cpu,7));
    printf("    Stack:\n");
    for (i = 8 ; i < 16; i += 2 ) {
	printf("\t%04.4x: %02.2x %02.2x\n", i, cpu->ram[i], cpu->ram[i+1]);
    }
}

#define GYRUSS_CPU_ROM (4096)
#define GYRUSS_CPU_ROMFILE "../roms/gy-13.bin"

byte GyrussDRAM;
long CyclesPerSample;

byte GyrussLatch(MCS48 *cpu, byte addr, int val)
{
    if (val > 0xFF) return GyrussDRAM;
    return 0;
}

MCS48 cpu;

void Sigint(int sig)
{
    printf("**BREAK**");
    signal(SIGINT,Sigint);
    cpu.running = 0;	/* stop virtual CPU */
}


void main(int argc, char **argv)
{
    int cycles;
    int done;
    long rate;

    if (M48Init(&cpu, MCS48_8039, GYRUSS_CPU_ROM) < 0) {
	printf("can't init cpu.\n");
	exit(1);
    }
    if (M48LoadRom(&cpu,GYRUSS_CPU_ROMFILE, 0, -1) < 0) {
	printf("Error loading romfile %s.\n",GYRUSS_CPU_ROMFILE);
	exit(1);
    }

    /* reset cpu */
    M48Reset(&cpu);
    /*
    M48SetPortHandler(&cpu,M48_PORT1,GyrussPort1);
    */
    cpu.xram = GyrussLatch;

    cpu.running = 1;

#define SAMPLE_RATE (22000)
#define CYCLES_PER_SAMPLE(rate) ((long)(((1.0 / ((double)(rate))) * 1000000.0) / 1.88))
    rate = SAMPLE_RATE;
    CyclesPerSample = CYCLES_PER_SAMPLE(rate);
    printf ("%d cycles per sample (%d hz output rate)\n",CyclesPerSample,rate);

    printf("Running preamble.\n");
    signal(SIGINT,Sigint);
    while (cpu.pc != 0x19 && cpu.running) M48Run(&cpu);
    printf("Preamble finish; %d cycles\n",cpu.cycles);

    done = 0;

    do {
	char buf[80], *p;
	int cnt, i;

	printf("%04.4x: ",cpu.pc); fflush(stdout);
	cnt = M48DAsm(buf,&(cpu.rom[cpu.pc]));
	for (i = 0; i < 3; i++) {
	    if (i < cnt)
		printf("%02.2x ",cpu.rom[cpu.pc+i]);
	    else
	        printf("   ");
	}
	printf("%-20.20s   # A=%02.2x, R0=%02.2x, R1=%02.2x\n",
		buf,cpu.a,M48REG(&cpu,0), M48REG(&cpu,1));

	printf("%sMCS48>", cpu.running ? "" : "(HALTED)"); fflush(stdout);

	if (isatty(fileno(stdin))) {
	    TMonGets(buf, sizeof(buf)-1);
	    printf("\n");
	    p = buf; 
	} else {
	    p = fgets(buf,sizeof(buf),stdin);
	    if (p == NULL) break;
	}
	cycles = 1;
	if (isdigit(*p)) {
	    long startms, endms;
	    cycles = atoi(p);

#ifdef PERF_TIME
	    startms = timeGetTime();
#endif
	    for (i  = 0 ; i < cycles && cpu.running ; i++ ) {
		M48Run(&cpu);
	    }
#ifdef PERF_TIME
	    endms = timeGetTime();

	    if (startms != endms)
		printf("%d instructions in %d ms;  %d instructions per second\n",
		    cycles, endms - startms, cycles * 1000 / (endms - startms));
#endif
	}
	switch (*p) {
	    case '\0':
		    M48Run(&cpu);	/* step CPU */
		    cpu.in = 0;		/* reset interrupt (temp) */
		    break;
	    case 'i': cpu.in = 1; break;	/* set interrupt */
	    case 'q': done = 1; break;
	    case 'r': M48DumpRegs(&cpu); break;
	    case 'g': { int last = cpu.pc;
		        do { M48Run(&cpu); } while (cpu.pc != last && cpu.running);
		      printf("Stopped at: $%04.4X.\n",cpu.pc);
	    		}
		      break;
	    case 'p':
		      GyrussDRAM = strtol(++p,0,0);
		      cpu.in = 1;
		      printf("Writing %x (%d) to LATCH and setting interrupt.\n",
			  GyrussDRAM, GyrussDRAM);
		      break;
	    case 'P': {
	    	      int last = cpu.pc;
		      GyrussDRAM = strtol(++p,0,0);
		      cpu.in = 1;
		      printf("Writing %x (%d) to LATCH and setting interrupt.\n",
			  GyrussDRAM, GyrussDRAM);
		      printf("Running effect.\n");
		      do { M48Run(&cpu); } while (cpu.pc != last && cpu.running);
		      printf("Stopped at: $%04.4X.\n",cpu.pc);
	    	      }
		      break;
	    case 's': { int newrate;
			++p;
		 	if ( *p >= '0' && *p <= '9' ) {
			    newrate = strtol(p,0,0);
			    if (newrate < 8000) newrate = 8000;
			    if (newrate > 88000) newrate = 88000;
			    rate = newrate;
			} 
			CyclesPerSample = CYCLES_PER_SAMPLE(rate);
			printf("%d cycles per sample (%d hz output rate)\n",
			    CyclesPerSample,rate);
		      }
		      break;
	}
    } while (!done);
    exit(0);
}

/*
** Change Log
** ----------
** $Log:$
**
*/

#define iprintf printf
#define igetc getch
void TMonBeep () { ; }


#define CTRL(a) ((a) - 'A' + 1)

extern int TMonGets(char *buf, int max)
{
    char c;
    int len = 0;

    while (1) {
	while ((c = igetc()) == -1);
	switch(c) {
	    case '\r':
	    case '\n':
		buf[len] = '\0';
		return(len);

	    case CTRL('?'):
	    case CTRL('H'):	/* backspace */
	        if (len > 0) {
		    len--;
		    iprintf("\b \b");
		}
		break;

	    case CTRL(']'):	/* escape */
	    case CTRL('U'):
	    case CTRL('X'):	/* kill line */
	        while (len > 0) {
		    len--;
		    iprintf("\b \b");
		}
		TMonBeep();
		break;
	    case CTRL('W'):	/* erase word/whitespace */
		if (len > 0) {
		    len--;
		    iprintf("\b \b");
		    if (buf[len] == ' ') {
			while (len > 0) {
			    if (buf[len-1] != ' ') break;
			    len--;
			    iprintf("\b \b");
			}
		    }
		    while (len > 0) {
			if (buf[len-1] == ' ') break;
			len--;
			iprintf("\b \b");
		    }
		    buf[len] = '\0';
		} else { TMonBeep(); }
		break;
	    default:
	        if (len == max || c < ' ' || c > '~') {
		    TMonBeep();
		    break;
		}
		buf[len++] = c; iprintf("%c",c);
		buf[len] = '\0';
		break;
	}
    }
}

/*
** Change Log
** ----------
** $Log: tmutil.c $
** Revision 1.1  1996/11/21 23:49:02  mcuddy
** Initial revision
**
*/
