/*
** $Id:$
**
** File: dis48.c -- disassembler for 8048/8039 microcontroller
**
** NOTE: this file is '#include'd by cpu48.c with the symbol DIS48_INCLUDE
** 	defined.
**
** This file is Copyright 1996 Michael Cuddy, Fen's Ende Sofware.
** Redistribution is allowed in source and binary form as long as both
** forms are distributed together with the file 'README'.  This copyright
** notice must also accompany the files.
**
** This software should be considered a small token to all of the
** emulator authors for thier dilligence in preserving our Arcade and
** Computer history.
**
** Michael Cuddy, Fen's Ende Software.
** 11/25/1996
**
*/
#include <stdio.h>
#include <string.h>

#ifndef DIS48_INCLUDE
/*
** stand-alone disassembler
*/
#include "mcs48.h"
#define FMT(a,b,c) a, b
#define PTRS_PER_FORMAT 2

#else
/*
** part of MCS48 emulator
*/
#define FMT(a,b,c) a, b, (char *)c
#define PTRS_PER_FORMAT 3

/* forward declarations of opcode emulation functions */
#include "ops48.h"

#endif	/* DIS48_INCLUDE */

char *Formats[] = {
    FMT("00000011dddddddd", "add a,#%D", _M48ADDID),
    FMT("01101rrr", "add a,%R", _M48ADDREG),
    FMT("0110000r", "add a,@%R", _M48ADDM),
    FMT("00010011dddddddd", "adc a,#%D", _M48ADDC),
    FMT("01111rrr", "adc a,%R", _M48ADDRGC),
    FMT("0111000r", "adc a,@%R", _M48ADDMC),
    FMT("01010011dddddddd", "anl a,#%D", _M48ANDID),
    FMT("01011rrr", "anl a,%R", _M48ANDREG),
    FMT("0101000r", "anl a,@%R", _M48ANDMM),
    FMT("10011000dddddddd", "anl bus,#%D", _M48ANDBUS),
    FMT("10011001dddddddd", "anl P1,#%D", _M48ANDBUS),
    FMT("10011010dddddddd", "anl P2,#%D", _M48ANDBUS),
    FMT("100111pp", "anld %P,a", _M48ANDP4),
    FMT("aaa10100aaaaaaaa", "call %A", _M48CALL),
    FMT("00100111", "clr a", _M48CLRA),
    FMT("10010111", "clr c", _M48CLRC),
    FMT("10100101", "clr f1", _M48CLRF1),
    FMT("10000101", "clr f0", _M48CLRF0),
    FMT("00110111", "cpl a", _M48CPLA),
    FMT("10100111", "cpl c", _M48CPLC),
    FMT("10010101", "cpl f0", _M48CPLF0),
    FMT("10110101", "cpl f1", _M48CPLF1),
    FMT("01010111", "da  a", _M48DAA),
    FMT("00000111", "dec a", _M48DECA),
    FMT("11001rrr", "dec %R", _M48DECR),
    FMT("00010101", "dis i", _M48DISI),
    FMT("00110101", "dis tcnti", _M48DISTI),
    FMT("11101rrraaaaaaaa", "djnz %R,%A", _M48DJNZ),
    FMT("00000101", "en  i", _M48ENI),
    FMT("00100101", "en  tcnti", _M48ENTI),
    FMT("01110101", "ent0 clk", _M48ENT0),
    FMT("00001001", "in  a,P0", _M48INP),
    FMT("00001010", "in  a,P1", _M48INP),
    FMT("00010111", "inc a", _M48INCA),
    FMT("00011rrr", "inc %R", _M48INCR),
    FMT("0001000r", "inc @%R", _M48INCM),
    FMT("00001000", "ins a,bus", _M48INSA),
    FMT("0001 0110aaaaaaaa", "jtf %A", _M48JMPIF),
    FMT("0010 0110aaaaaaaa", "jnt0 %A", _M48JMPIF),
    FMT("0011 0110aaaaaaaa", "jt0 %A", _M48JMPIF),
    FMT("0100 0110aaaaaaaa", "jnt1 %A", _M48JMPIF),
    FMT("0101 0110aaaaaaaa", "jt1 %A", _M48JMPIF),
    FMT("0111 0110aaaaaaaa", "jf1 %A", _M48JMPIF),
    FMT("1000 0110aaaaaaaa", "jni %A", _M48JMPIF),
    FMT("1001 0110aaaaaaaa", "jnz %A", _M48JMPIF),
    FMT("1011 0110aaaaaaaa", "jf0 %A", _M48JMPIF),
    FMT("1100 0110aaaaaaaa", "jz  %A", _M48JMPIF),
    FMT("1110 0110aaaaaaaa", "jnc %A", _M48JMPIF),
    FMT("1111 0110aaaaaaaa", "jc  %A", _M48JMPIF),
    FMT("bbb10010aaaaaaaa", "jb%B %A", _M48JBB),
    FMT("aaa00100aaaaaaaa", "jmp %A", _M48JMP),
    FMT("10110011", "jmpp @a", _M48JMPP),
    FMT("00100011dddddddd", "mov a,#%D", _M48MOVAID),
    FMT("11111rrr", "mov a,%R", _M48MOVAR),
    FMT("1111000r", "mov a,@%R", _M48MOVAM),
    FMT("11000111", "mov a,psw", _M48MOVAPSW),
    FMT("10111rrrdddddddd", "mov %R,#%D", _M48MOVRID),
    FMT("10101rrr", "mov %R,a", _M48MOVRA),
    FMT("1010000r", "mov @%R,a", _M48MOVMA),
    FMT("1011000rdddddddd", "mov @%R,#%D", _M48MOVMID),
    FMT("11010111", "mov psw,a", _M48MOVPSWA),
    FMT("000011pp", "movd a,%P", _M48MOVDAP),
    FMT("001111pp", "movd %P,a", _M48MOVDPA),
    FMT("01000010", "mov a,t", _M48MOVAT),
    FMT("01100010", "mov t,a", _M48MOVTA),
    FMT("11100011", "movp3 a,@a", _M48MOVP3),
    FMT("10100011", "movp a,@a", _M48MOVPAA),
    FMT("1000000r", "movx a,@%R", _M48MOVXAM),
    FMT("1001000r", "movx @%R,a", _M48MOVXMA),
    FMT("0100 1rrr", "orl a,%R", _M48ORLAR),
    FMT("0100 000r", "orl a,@%R", _M48ORLAM),
    FMT("0100 0011dddddddd", "orl a,#%D", _M48ORLAID),
    FMT("1000 1000dddddddd", "orl bus,#%D", _M48ORLP),
    FMT("1000 1001dddddddd", "orl P1,#%D", _M48ORLP),
    FMT("1000 1010dddddddd", "orl P2,#%D", _M48ORLP),
    FMT("1000 11pp", "orld %P,a", _M48ORLD),
    FMT("00000010", "outl bus,a", _M48OUTLBUS),
    FMT("001110pp", "outl %P,a", _M48OUTLP),
    FMT("10000011", "ret", _M48RET),
    FMT("10010011", "retr", _M48RETR),
    FMT("11100111", "rl  a", _M48RLA),
    FMT("11110111", "rlc a", _M48RLCA),
    FMT("01110111", "rr  a", _M48RRA),
    FMT("01100111", "rrc a", _M48RRCA),
    FMT("11100101", "sel mb0", _M48SELM0),
    FMT("11110101", "sel mb1", _M48SELM1),
    FMT("11000101", "sel rb0", _M48SELR0),
    FMT("11010101", "sel rb1", _M48SELR1),
    FMT("01100101", "stop tcnt", _M48STPTC),
    FMT("01000101", "strt tcnt", _M48STRTTC),
    FMT("01010101", "strt t", _M48STRTT),
    FMT("01000111", "swap a", _M48SWAPA),
    FMT("00101rrr", "xcha a,%R", _M48XCHAID),
    FMT("0010000r", "xcha a,@%R", _M48XCHAM),
    FMT("0011000r", "xchda a,@%R", _M48XCHADA),
    FMT("1101 0011dddddddd", "xrl a,#%D", _M48XRLAID),
    FMT("1101 1rrr", "xrl a,%R", _M48XRLR),
    FMT("1101 000r", "xrl a,@%R", _M48XRLM),
    FMT("00000000", "nop", _M48NOP),
    NULL
};
#define MAX_OPS (((sizeof(Formats) / sizeof(Formats[0])) - 1) / PTRS_PER_FORMAT)

typedef struct opcode {
    byte mask;	/* instruction mask */
    byte bits;	/* constant bits */
    char extcode;	/* value that gets extension code */
    char *parse;	/* how to parse bits */
    char *fmt;		/* instruction format */
#ifdef DIS48_INCLUDE
    M48OpcodeFunc func;
#endif
} M48Opcode;

M48Opcode Op[MAX_OPS+1];

void M48ParseOps()
{
    char *p, **ops;
    byte mask, bits;
    int bit;
    int i;

    ops = Formats; i = 0;
    while (*ops) {
	p = *ops;
	mask = 0; bits = 0; bit = 7;
	while (*p && bit >= 0) {
	    switch (*p++) {
		case '1': mask |= 1<<bit; bits |= 1<<bit; bit--; break;
		case '0': mask |= 1<<bit; bit--; break;
		case ' ': break;
		case 'b':
		case 'a': case 'r': case 'd': case 'p':
		    bit --;
		    break;
		default:
		    printf("Invalid instruction encoding '%s %s'\n",
			ops[0],ops[1]);
		    exit(1);
	    }
	}
	if (bit != -1 ) {
	    printf("not enough bits in encoding '%s %s' %d\n",
	    		ops[0],ops[1],bit);
	    exit(1);
	}
	while (isspace(*p)) p++;
	if (*p) Op[i].extcode = *p;
	Op[i].bits = bits;
	Op[i].mask = mask;
	Op[i].fmt = ops[1];
	Op[i].parse = ops[0];
#ifdef DIS48_INCLUDE
	Op[i].func = (M48OpcodeFunc)(ops[2]);
#endif

	/* printf("<%s> <%s> <%d> bits = %02.2x, mask = %02.2x\n",
		Op[i].fmt,Op[i].parse,Op[i].extcode,Op[i].bits,Op[i].mask); */
	ops += PTRS_PER_FORMAT;
	i++;
    }
}

int M48DAsm(char *str,byte *addr)
{
    int b, a, d, r, p;	/* these can all be filled in by parsing an instruction */
    int i;
    int op;
    int cnt = 1;
    int code, bit;
    char *cp;

    op = -1;	/* no matching opcode */
    for ( i = 0; i < MAX_OPS; i++) {
	if ((*addr & Op[i].mask) == Op[i].bits) {
	    if (op != -1) {
		printf("Error: opcode %02.2X matches %d (%s) and %d (%s)\n",
		    *addr,i,Op[i].fmt,op,Op[op].fmt);
	    }
	    op = i;
	}
    }
    if (op == -1) {
	sprintf(str,"db %02.2x",*addr);
	return cnt;
    }
    code = *addr;
    if (Op[op].extcode) {
	cnt++; addr++;
	code <<= 8;
	code |= *addr;
	bit = 15;
    } else {
	bit = 7;
    }
    /* printf("code = %x (%s, %s)\n",code,Op[op].parse, Op[op].fmt); */

    /* shift out operands */
    cp = Op[op].parse;
    b = a = d = r = p = 0;

    while (bit >= 0) {
	/* printf("{%c/%d}",*cp,bit); */
	switch(*cp) {
	    case 'a': a <<=1; a |= ((code & (1<<bit)) ? 1 : 0); bit--; break;
	    case 'b': b <<=1; b |= ((code & (1<<bit)) ? 1 : 0); bit--; break;
	    case 'd': d <<=1; d |= ((code & (1<<bit)) ? 1 : 0); bit--; break;
	    case 'r': r <<=1; r |= ((code & (1<<bit)) ? 1 : 0); bit--; break;
	    case 'p': p <<=1; p |= ((code & (1<<bit)) ? 1 : 0); bit--; break;
	    case ' ': break;
	    case '1': case '0':  bit--; break;
	    case '\0': printf("premature end of parse string, opcode %x, bit = %d\n",code,bit); exit(1);
	}
	cp++;
    }

    /* now traverse format string */
    cp = Op[op].fmt;
    while (*cp) {
	if (*cp == '%') {
	    char num[10], *q;
	    cp++;
	    switch (*cp++) {
		case 'A': sprintf(num,"$%04.4X",a); break;
		case 'B': sprintf(num,"%d",b); break;
		case 'D': sprintf(num,"%d",d); break;
		case 'R': sprintf(num,"r%d",r); break;
		case 'P': sprintf(num,"p%d",p); break;
		default:
		    printf("illegal esape character in format '%s'\n",Op[op].fmt);
		    exit(1);
	    }
	    q = num; while (*q) *str++ = *q++;
	    *str = '\0';
	} else {
	    *str++ = *cp++;
	    *str = '\0';
	}
    }
    return cnt;
}

#ifndef DIS48_INCLUDE	/* define to use this source file as a header ;-) */
int main(int argc,char *argv[])
{
  FILE *F;
  long Counter;
  int len, offset;
  int N,I;
  byte Buf[16];
  char S[128];

  M48ParseOps();

  if(argc<2)
  {
    puts("MCS-48 Disassembler 1.1 by Michael Cuddy, Fen's Ende Software (C)1996");
    puts("Usage: dis48 <input file> [ <start-addr> [ <len> ] ]");
    return(0);
  }

  if(!(F=fopen(argv[1],"rb")))
  { printf("\n%s: Can't open file %s\n",argv[0],argv[1]);return(1);
  }
      argv++; argc--;
  if (argv[1]) {
      offset = strtol(argv[1],NULL,0);
      printf("offset = %d (%s)\n",offset,argv[1]);
      argv++; argc--;
  } else {offset = 0; }
  if (argv[1]) {
      len = strtol(argv[1],NULL,0);
      printf("len = %d, %s\n",len,argv[1]);
      argv++; argc--;
  } else { len = 0x7FFFFFFF; }

  Counter=0L;N=0;
  if (fseek(F,offset,0) != 0) {
      fprintf(stderr,"Error seeking to offset %d\n",offset);
      exit(1);
  }
  Counter = offset;
  while(N+=fread(Buf+N,1,16-N,F))
  {
    int ii;
    if(N<16) memset(Buf+N,0,16-N);
    I=M48DAsm(S,Buf);
    printf("%08lX: ", Counter);
    for (ii = 0; ii < I; ii++) {
	printf("%02.2x ",Buf[ii]);
    }
    for (; ii < 4; ii++) { printf("   "); }
    printf("\t%s\n",S);
    Counter+=I;N-=I;
    len -= I; if (len <= 0) break;
    if(N) memcpy(Buf,Buf+16-N,N);
  }

  fclose(F);return(0);
}
#endif

/*
** Change Log
** ----------
** $Log:$
**
*/
