// ------------
//	graphics.c
// ------------

#include "bombjack.h"
#include "chars.h"
#include "background_chars.h"
#include "sprites.h"
#include "tilemaps.h"
#include "palette.h"

OAMEntry gSprites[128];			// local copy of OAM

static const u16 color_table[] =
{
	0x0f00, 0x0f22, 0x0f44, 0x0f66, 0x0f99, 0x0fbb, 0x0fdd, 0x0fff,
	0x0dff, 0x0bff, 0x09ff, 0x06ff, 0x04ff, 0x02ff, 0x00ff, 0x00df,
	0x00bf, 0x009f, 0x006f, 0x004f, 0x002f, 0x000f, 0x020f, 0x040f,
	0x060f, 0x090f, 0x0b0f, 0x0d0f, 0x0f0f, 0x0f0d, 0x0f0b, 0x0f09,
	0x0f06, 0x0f04, 0x0f02, 0x0f00, 0x0f22, 0x0f44, 0x0f66, 0x0f99,
	0x0fbb, 0x0fdd
};

static const u16 color_table2[] =
{
	0x000f, 0x020f, 0x040f, 0x060f, 0x090f, 0x0b0f, 0x0d0f, 0x0f0f,
	0x0f0d, 0x0f0b, 0x0f09, 0x0f06, 0x0f04, 0x0f02, 0x0f00, 0x0f20,
	0x0f40, 0x0f60, 0x0f90, 0x0fb0, 0x0fd0, 0x0ff0, 0x0df0, 0x0bf0,
	0x09f0, 0x06f0, 0x04f0, 0x02f0, 0x00f0, 0x00f2, 0x00f4, 0x00f6,
	0x00f9, 0x00fb, 0x00fd, 0x00ff, 0x00df, 0x00bf, 0x009f, 0x006f,
	0x004f, 0x002f, 0x000f, 0x020f, 0x040f, 0x060f, 0x090f, 0x0b0f,
};

static const u16 bkg2_palette[] =
{
	0x0000, 0x0b00, 0x0034, 0x0156, 0x0378, 0x059a, 0x07cc, 0x09ef
};

// -------------------------------------------------------------------------------------------
//
//	void rotateColors(void)
//
//	Rotates palettes 14 and 15 for flashy effects :)
//
// -------------------------------------------------------------------------------------------

void rotateColors(void)
{
	static u8 count = 0;
	static u8 color_count = 0;
	static u8 color_count2 = 0;

	int i;
	
	if(count++ == 5)
	{
		for(i = 0; i < 7; i++)
			setPaletteColor(15, i + 1, color_table[color_count + i]);

		color_count--;
		if(color_count == 255) color_count = 35;

		for(i = 0; i < 7; i++)
			setPaletteColor(14, i + 1, color_table2[color_count2 + i]);

		color_count2--;
		if(color_count2 == 255) color_count2 = 41;
		
		count = 0;
	}

} // of rotateColors()

// -------------------------------------------------------------------------------------------
//
//	void setPalette(u8 palette_num, u8 palettes, u8 *palette_base)
//
//	Copies in a palette at the specified location.
//
// -------------------------------------------------------------------------------------------

void setPalette(u8 palette_num, u8 palettes, u16 *palette_base)
{
	int i;
	int offset = palette_num << 4;
	
	for(i = 0; i < palettes; i++)
	{
		int j;

		for(j = 0; j < 8; j++)
		{
			u8 red = palette_base[(i << 3) + j] & 0x000f;
			u8 green = (palette_base[(i << 3) + j] & 0x00f0) >> 4;
			u8 blue = (palette_base[(i << 3) + j] & 0x0f00) >> 8;
		
			// expand out from 4-bit to 5-bit
			red =   (red << 1)   | ((red & 0x8) >> 3);
			green = (green << 1) | ((green & 0x8) >> 3);
			blue =  (blue << 1)  | ((blue & 0x8) >> 3);
			
			BG_PALETTE_MEM[(i << 4) + j + offset] = RGB(red, green, blue);
			OBJ_PALETTE_MEM[(i << 4) + j + offset] = RGB(red, green, blue);
		}
	}

} // of setPalette()

// -------------------------------------------------------------------------------------------
//
//	void setPaletteColor(u8 palette_num, u8 color_num, u16 palette_color)
//
//	Sets the specified color in the specified palette -- palette_color is a 12-bit color, not
//	the native GBA 15-bit color.
//	
// -------------------------------------------------------------------------------------------

void setPaletteColor(u8 palette_num, u8 color_num, u16 palette_color)
{
	u8 red = palette_color & 0x000f;
	u8 green = (palette_color & 0x00f0) >> 4;
	u8 blue = (palette_color & 0x0f00) >> 8;
	
	// expand from 4-bits to 5-b7it
	red =   (red << 1)   | ((red & 0x8) >> 3);
	green = (green << 1) | ((green & 0x8) >> 3);
	blue =  (blue << 1)  | ((blue & 0x8) >> 3);

	BG_PALETTE_MEM[(palette_num << 4) + color_num] = RGB(red, green, blue);
	OBJ_PALETTE_MEM[(palette_num << 4) + color_num] = RGB(red, green, blue);
	
} // of setPaletteColor()

// -------------------------------------------------------------------------------------------
//
//	void setBackground(u32 background);
//
//	Copies the tileset for the specified background to VRAM.
//	
// -------------------------------------------------------------------------------------------

void setBackground(u32 background)
{
	static u32 current_background = 5;		// default to the blank background
	int x,y;
	u16 *odd_ptr = kBG3Map;
	u16 *even_ptr = (u16*)(kBG3Map + 32);
	
	// make sure that the background stays in the proper range
	background &= 0x7;
	
	// cheap hack to display background number
//	kBG0Map[0] = '0' + background;
	
	// background 6 and 7 are the same
	if(background == 7) background = 6;
	
	// no need to copy it again
	if(background == current_background) return;
	
	// we need to copy over the correct tilemap for the specified background
	// each byte in the tilemap represents a 2x2 square
	for(x = 0; x < 16; x++)
	{
		for(y = 15; y >= 0; y--)
		{
			u16 value = tilemap[(background << 1)][y * 16 + x] << 2;
			u8 attribute = tilemap[(background << 1) + 1][y * 16 + x];
			u16 palette = (attribute & 0xf) << 12;
			
			if(attribute & 0x80)
			{
				*odd_ptr++ = (value) | 0x0400 | palette;
				*odd_ptr++ = (value + 2) | 0x0400 | palette;

				*even_ptr++ = (value + 1) | 0x0400 | palette;
				*even_ptr++ = (value + 3) | 0x0400 | palette;
			}

			else
			{
				*odd_ptr++ = (value + 2) | palette;
				*odd_ptr++ = (value) | palette; 
			
				*even_ptr++ = (value + 3) | palette;
				*even_ptr++ = (value + 1) | palette;
			}
		}

		odd_ptr += 32;
		even_ptr += 32;
	}
	
	current_background = background;

	// set the appropriate palettes for this background
	if(background == 2)
		setPalette(10, 1, (u16*)bkg2_palette);

} // of setBackground()

// -------------------------------------------------------------------------------------------
//
//	void loadTiles(u32 *vram_ptr, u8 *tiles, u16 num_tiles);
//
//	The three bitplanes of the tiles are stored separately.  Also, the tile data is rotated.
//	This function recombines and rotates the data back before setting the tile in VRAM
//
//	Eventually, I may convert the data to native GBA format, but using it this way is 25%
//	smaller due to the fact that the original data doesn't have a fourth bitplane.
//	
// -------------------------------------------------------------------------------------------

static void loadTiles(u32 *vram_ptr, u8 *tiles, u16 num_tiles)
{
	u8	*plane1_ptr = (u8*)&tiles[16];			// 8 bytes for each plane
	u8	*plane2_ptr = (u8*)&tiles[8];
	u8	*plane3_ptr = (u8*)&tiles[0];
	u32	i;

	for(i = 0; i < num_tiles; i++)
	{
		u32 row1 = 0;
		u32 row2 = 0;
		u32 row3 = 0;
		u32 row4 = 0;
		u32 row5 = 0;
		u32 row6 = 0;
		u32 row7 = 0;
		u32 row8 = 0;
		
		u32 j;
		
		for(j = 0; j < 8; j++)
		{
			u8	plane1 = *plane1_ptr++;
			u8	plane2 = *plane2_ptr++;
			u8	plane3 = *plane3_ptr++;
	
			row1 <<= 4; row2 <<= 4; row3 <<= 4; row4 <<=4;
			row5 <<= 4; row6 <<= 4; row7 <<= 4; row8 <<=4;
			
			row1 |= ((plane1 & 0x1) | ((plane2 & 0x1) << 1) | ((plane3 & 0x1) << 2));
			plane1 >>= 1; plane2 >>= 1; plane3 >>= 1;
		
			row2 |= ((plane1 & 0x1) | ((plane2 & 0x1) << 1) | ((plane3 & 0x1) << 2));
			plane1 >>= 1; plane2 >>= 1; plane3 >>= 1;
			
			row3 |= ((plane1 & 0x1) | ((plane2 & 0x1) << 1) | ((plane3 & 0x1) << 2));
			plane1 >>= 1; plane2 >>= 1; plane3 >>= 1;
		
			row4 |= ((plane1 & 0x1) | ((plane2 & 0x1) << 1) | ((plane3 & 0x1) << 2));
			plane1 >>= 1; plane2 >>= 1; plane3 >>= 1;

			row5 |= ((plane1 & 0x1) | ((plane2 & 0x1) << 1) | ((plane3 & 0x1) << 2));
			plane1 >>= 1; plane2 >>= 1; plane3 >>= 1;
		
			row6 |= ((plane1 & 0x1) | ((plane2 & 0x1) << 1) | ((plane3 & 0x1) << 2));
			plane1 >>= 1; plane2 >>= 1; plane3 >>= 1;
		
			row7 |= ((plane1 & 0x1) | ((plane2 & 0x1) << 1) | ((plane3 & 0x1) << 2));
			plane1 >>= 1; plane2 >>= 1; plane3 >>= 1;
		
			row8 |= ((plane1 & 0x1) | ((plane2 & 0x1) << 1) | ((plane3 & 0x1) << 2));
		}
		
		*vram_ptr++ = row8; *vram_ptr++ = row7; *vram_ptr++ = row6; *vram_ptr++ = row5;
		*vram_ptr++ = row4; *vram_ptr++ = row3; *vram_ptr++ = row2; *vram_ptr++ = row1;

		// skip over the other two planes
		plane1_ptr += 16;
		plane2_ptr += 16;
		plane3_ptr += 16;
	}

} // of loadTiles()

// -------------------------------------------------------------------------------------------
//
//	void hideSprites(void)
//
//	Moves all sprites in OAM to offscreen so there isn't any garbage on the screen.
//	
// -------------------------------------------------------------------------------------------

void hideSprites(void)
{
	int i;

	for(i = 0; i < 128; i++)
		gSprites[i].attribute1 = 240;
	
} // of hideSprites()

// -------------------------------------------------------------------------------------------
//
//	void setSpriteTile(u8 sprite, u8 tile, u8 flag32)
//
//	Sets the specified sprite tile -- in this case, tile number is the original tile number,
//	not the GBA tile number.  If the flag is set, the sprite will be a 32x32 sprite, not the
//	16x16.
//
// -------------------------------------------------------------------------------------------

void setSpriteTile(u8 sprite, u8 tile, u8 flag32)
{
	gSprites[sprite].attribute2 &= 0xfc00;		// clear out the old tile number
	gSprites[sprite].attribute1 &= 0x01ff;		// reset the size attribute as well

	if(flag32)
	{
		gSprites[sprite].attribute1 |= SPRITE_SIZE_32;
		gSprites[sprite].attribute2 |= 512 + (tile << 4);
	}

	else
	{
		gSprites[sprite].attribute1 |= SPRITE_SIZE_16; 
		gSprites[sprite].attribute2 |= (tile << 2);
	}

} // of setSpriteTile()

// -------------------------------------------------------------------------------------------
//
//	void drawString(u16 *map, const char *string, u8 x, u8 y, u16 palette)
//
//	Draws a string to the specified 32x32 tile map.
//
// -------------------------------------------------------------------------------------------

void drawString(u16 *map, const char *string, u8 x, u8 y, u16 palette)
{
	map += (y * 32) + x;
	palette <<= 12;
	
	while(*string != '\0') *map++ = (*string++) | palette;

} // of drawString()

// -------------------------------------------------------------------------------------------
//
//	void drawBCD(u16 *map, u32, num, u8 x, u8 y, u16 palette)
//
//	Draws a BCD number to the specified map
//
// -------------------------------------------------------------------------------------------

void drawBCD(u16 *map, u32 num, u8 x, u8 y, u16 palette)
{
	u8 non_zero_flag = 0;
	int i;
	
	map += (y * 32) + x;
	palette <<= 12;
	
	for(i = 0; i < 8; i++)
	{
		u8 value = (num & 0xf0000000) >> 28;
		num <<= 4;

		if((value > 0) || (non_zero_flag))
		{
			non_zero_flag = 1;
			*map++ = ('0' + value) | palette;
		}

		else map++;
	}

} // of drawBCD()

// -------------------------------------------------------------------------------------------
//
//	void initializeGraphics(void)
//
//	Sets all necessary registers and copies in tilesets.
//	
// -------------------------------------------------------------------------------------------

void initializeGraphics(void)
{
	u32 zero = 0;
	
	// turn off the display for now
	REG_DISPCNT = DISP_LCDC_OFF | DISP_MODE_0;
	
	// clear out VRAM
	REG_DM3SAD = (u32)&zero;
	REG_DM3DAD = (u32)VRAM;
	REG_DM3CNT = 0x6000 | DMA_SOURCE_FIXED | DMA_32NOW;
	
	// load up all of our tilesets
	loadTiles(kCharTileset, (u8*)chars, kNumCharTiles);
	loadTiles(kBackgroundTileset, (u8*)background_tiles, kNumBackgroundTiles);
	loadTiles(kSpriteTileset, (u8*)sprite_tiles, kNumSpriteTiles);
	
	// copy in the main palettes 
	setPalette(0, 16, (u16*)main_palette);
	rotateColors();

	setBackground(5);

	// initialize sprites
	hideSprites();

	// setup our display registers
	REG_BG0CNT = BGCNT_TILEMAP_C000 | BGCNT_TILEDATA_0000 | BGCNT_PRIORITY_0;
	REG_BG1CNT = BGCNT_TILEMAP_C800 | BGCNT_TILEDATA_0000 | BGCNT_PRIORITY_1;
	REG_BG3CNT = BGCNT_TILEMAP_D000 | BGCNT_TILEDATA_4000 | BGCNT_PRIORITY_3;

	REG_BG3HOFS = 8;		// this shouldn't change during the game
	REG_BG3VOFS = 16;		// but this will

	// window used while in game mode
	REG_WIN0H = 0x08e7;
	REG_WIN0V = 0x108f;

	// BG0, BG3
	REG_WININ = WIN0_BG0 | WIN0_BG3 | WIN0_SPRITES;
	REG_WINOUT = WIN0_BG1;
	
	REG_DISPCNT = DISP_MODE_0 | DISP_BG0_ON | DISP_BG1_ON | DISP_BG3_ON | DISP_OBJ_ON | DISP_OM_1D;
	
} // of initializeGraphics()
