#include "bombjack.h"

#include "boards.h"

// ----------------------------
//	Character Tile Definitions
// ----------------------------

#define kPlayerTile				12
#define kXTile					192
#define kLeftBracketTile		194
#define kBig1Tile				196

// -------------------------
//	Sprite Tile Definitions
// -------------------------

#define kPlayerSittingSprite	24
#define kStartLeftSprite		64
#define kStartRightSprite		68

// ------------------
//	Game Mode Status
// ------------------

#define kPreStartGameMode		0
#define kStartGameMode			1
#define kActionGameMode			2

// ------------
//	Game State
// ------------

typedef struct
{
	u8 round;
	u16 score;		// BCD format
	u8 lives;

	u8 x_loc;
	u8 y_loc;
	
} gameState;

static gameState states[2];
static u8 gameModeStatus = kPreStartGameMode;
static u8 currentPlayer = 0;

#define scrollBackground(y)		{ \
									REG_BG0VOFS = (y); \
									REG_BG3VOFS = (y); \
								}

// ------------------------------------------------------------------------------------------
//	
//	void initGameState(void)
//
//	Initializes the game state for a new game.
//	
// ------------------------------------------------------------------------------------------

static void initGameState(void)
{
	int i;
	
	currentPlayer = 0;
	
	for(i = 0; i < 1; i++)
	{
		states[i].x_loc = 111;
		states[i].y_loc = 111;
		states[i].round = 1;
		states[i].lives = 3;
		states[i].score = 0;
	}

} // of initGameState()

// ------------------------------------------------------------------------------------------
//
//	void drawPlayers(u8 players)
//
//	Draws the remaining players in the bottom left corner of the screen
//
// ------------------------------------------------------------------------------------------

static void drawPlayers(u8 players)
{
	int i;
	
	for(i = 0; i < players; i++)
		draw2x2TileBG1(kPlayerTile, 1 + (i * 2), 18, 0);
	
} // of drawPlayers()

// ------------------------------------------------------------------------------------------
//
//	void drawBoard(u8 board)
//
//	Draws the board to the screen.
//
// ------------------------------------------------------------------------------------------

static void drawBoard(u8 board)
{
	switch(board)
	{
		case 0:
			setBackground(2);
			setPalette(9, 1, (u16*)&board_palette[0][0]);
			setPalette(5, 1, (u16*)&board_palette[1][0]);
			REG_DM3SAD = (u32)&boards[0][0];
			break;

		default:		// shouldn't happen
			return;
			break;
	}

	REG_DM3DAD = (u32)&kBG0Map[32 * 2];
	REG_DM3CNT = 448 | DMA_32NOW;
	
} // of drawBoard()

// ------------------------------------------------------------------------------------------
// ------------------------------------------------------------------------------------------

void initGameMode(u8 players)
{
	clearBG0();
	clearBG1();
	hideSprites();

	// we use BG1 for the score area outside the playing area
	REG_BG1HOFS = 0;
	REG_BG0HOFS = 8;

	scrollBackground(56);
	
	REG_DISPCNT = REG_DISPCNT | DISP_W0_ON;

	drawBG1String("SIDE-ONE", 2, 0, 15);
	drawBG1String("0", 9, 1, 5);

	if(players == 2)
	{
		drawBG1String("SIDE-TWO", 20, 0, 5);
		drawBG1String("0", 27, 1, 5);
	}

	draw1x2TileBG1(kLeftBracketTile, 13, 0, 14);
	draw1x2TileBG1(kXTile, 14, 0, 14);
	draw1x2TileBG1(198, 15, 0, 14);
	draw1x2TileBG1(kBig1Tile, 16, 0, 14);

	drawBG1String("ROUND", 15, 18, 1);
	drawBG1String("-1-", 16, 19, 3);
	drawBG1String("HI-SCORE", 21, 18, 0);
	drawBG1BCD(gHighScores[0].score, 20, 19, 5);
	drawBG1String("0", 28, 19, 5);

	drawPlayers(3);

	// draw the playing area
	drawBoard(0);

	gameModeStatus = kPreStartGameMode;

} // of initGameMode()

// ------------------------------------------------------------------------------------------
//
//	void animateStartMode(u8 start_flag)
//
//	Handles animating the "Start" sprite that occurs at the beginning of a turn/round.
//
// ------------------------------------------------------------------------------------------

static void animateStartMode(u8 start_flag)
{
	static u16 x = 0;		// x position/timer

	// we need to initialize the start sprites
	if(start_flag)
	{
		int i;
		
		x = 0;

		for(i = 0; i < 8; i++)
		{
			setSpriteTile(i, kStartLeftSprite + i, FALSE);
			setSpritePalette(i, 11);
			setSpriteYLoc(i, 72);
		}

		setPaletteColor(11, 1, 0x0000);
		setPaletteColor(11, 2, 0x0fff);
	}

	else x+= 2;

	// reposition the sprites
	if(x < 145)
	{
		setSpriteXLoc(0, x - 56);
		setSpriteXLoc(1, x - 40);
		setSpriteXLoc(2, x - 24);
		setSpriteXLoc(3, x - 8);

		setSpriteXLoc(4, 232 - x);
		setSpriteXLoc(5, 248 - x);
		setSpriteXLoc(6, 264 - x);
		setSpriteXLoc(7, 280 - x);
	}

	// check to see if we should leave the start mode
	else if (x == 250)
	{
		int i;
		
		for(i = 0; i < 8; i++)
			setSpriteTile(i, 0, FALSE);

		setSpriteTile(0, kPlayerSittingSprite, FALSE);
		setSpritePalette(0, 0);
		setSpriteXLoc(0, states[currentPlayer].x_loc);
		setSpriteYLoc(0, 72);
		
		gameModeStatus = kActionGameMode;
	}

} // of animateStartMode()

// ------------------------------------------------------------------------------------------
//
//	void handleAction(void)
//
//	Handles the movement of the player.
//	
// ------------------------------------------------------------------------------------------

static void handleAction(void)
{
	u8 y_loc = states[currentPlayer].y_loc;
	u8 x_loc = states[currentPlayer].x_loc;

	// update the background position for the previous frame
	if(y_loc <= 56)
	{
		scrollBackground(0);
	}

	else if(y_loc >= 152)
	{
		scrollBackground(97);
	}

	else
	{
		scrollBackground(y_loc - 55);
	}

	// check the keys, and see if we need to move
	if((gKeys & KEY_U) && (y_loc > 0))
	{
		y_loc--;
	}

	if((gKeys & KEY_D) && (y_loc < 207))
	{
		y_loc++;
	}

	if((gKeys & KEY_L) && (x_loc > 8))
	{
		x_loc--;
	}
	
	if((gKeys & KEY_R) && (x_loc < 216))
	{
		x_loc++;
	}
	
	// handle the sprite movement here, since the sprite update won't happen until the next
	// frame anyway.  We handle the background scrolling at that time.
	
	setSpriteXLoc(0, x_loc);

	if(y_loc <= 56)
	{
		setSpriteYLoc(0, y_loc + 16);
	}

	else if(y_loc >= 152)
	{
		setSpriteYLoc(0, y_loc - 80);
	}

	else
	{
		setSpriteYLoc(0, 72);
	}

	// update the gamestate
	states[currentPlayer].x_loc = x_loc;
	states[currentPlayer].y_loc = y_loc;
	
} // of handleAction()

// ------------------------------------------------------------------------------------------
//
//	void updateGameMode(void)
//
//	Called by the main loop to update the game state every frame.
//
// ------------------------------------------------------------------------------------------

void updateGameMode(void)
{
	switch(gameModeStatus)
	{
		case kPreStartGameMode:
			initGameState();
			animateStartMode(TRUE);
			gameModeStatus = kStartGameMode;
			break;

		case kStartGameMode:
			animateStartMode(FALSE);
			break;

		case kActionGameMode:
			handleAction();
			break;
	}
	
} // of updateGameMode()
